package net.aihelp.data.localize;

import android.text.TextUtils;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;

import net.aihelp.data.localize.config.InitBusinessHelper;
import net.aihelp.data.localize.config.ProcessEntranceHelper;
import net.aihelp.data.localize.config.StyleSheetHelper;
import net.aihelp.data.localize.config.CommonTextHelper;
import net.aihelp.data.localize.config.TemplateBusinessHelper;
import net.aihelp.data.localize.config.UploadLimitHelper;
import net.aihelp.data.localize.data.FaqHelper;
import net.aihelp.data.localize.data.LocaleStringHelper;
import net.aihelp.data.localize.util.LocalizeUtil;
import net.aihelp.data.track.resource.ResourceTracker;
import net.aihelp.init.InitHelper;
import net.aihelp.utils.FileUtil;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import androidx.annotation.RestrictTo;


@RestrictTo(RestrictTo.Scope.LIBRARY)
public class LocalizeHelper {

    public static final int FETCH_SOURCE_INIT = 1;
    public static final int FETCH_SOURCE_TEMPLATE = 2;

    public static final int MODE_FAQ_SECTION = 1;

    public static final int MODE_INIT_LOCALE = 11;
    public static final int MODE_INIT_PROCESS = 12;
    public static final int MODE_INIT_UPLOAD_LIMIT = 13;
    public static final int MODE_INIT_TEXT = 14;
    public static final int MODE_INIT_BUSINESS_LOGIC = 15;

    public static final int MODE_TEMPLATE_STYLE_SHEET = 21;
    public static final int MODE_TEMPLATE_BUSINESS_LOGIC = 22;
    public static final int MODE_TEMPLATE_FAQ_HOT_TOPIC = 23;
    public static final int MODE_TEMPLATE_TEXT = 24;

    public static void resetLocalizeData() {
        FaqHelper.INSTANCE.reset();
        LocaleStringHelper.INSTANCE.reset();
        CommonTextHelper.INSTANCE.reset();
        ProcessEntranceHelper.INSTANCE.reset();
        UploadLimitHelper.INSTANCE.reset();
    }

    private static List<Integer> getModeList(int fetchSource) {
        ArrayList<Integer> modeList = new ArrayList<>();
        if (fetchSource == FETCH_SOURCE_INIT) {
            if (Const.TOGGLE_LOCALIZE_VIA_INIT || LocalizeUtil.isAlreadyLocalized(MODE_FAQ_SECTION)) {
                modeList.add(MODE_FAQ_SECTION);
            }
            modeList.add(MODE_INIT_LOCALE);
            modeList.add(MODE_INIT_TEXT);
            modeList.add(MODE_INIT_UPLOAD_LIMIT);
            modeList.add(MODE_INIT_BUSINESS_LOGIC);
            modeList.add(MODE_INIT_PROCESS);
        }

        if (fetchSource == FETCH_SOURCE_TEMPLATE) {
            if (!LocalizeUtil.isAlreadyLocalized(MODE_FAQ_SECTION)) {
                modeList.add(MODE_FAQ_SECTION);
            }
            modeList.add(MODE_TEMPLATE_FAQ_HOT_TOPIC);
            modeList.add(MODE_TEMPLATE_BUSINESS_LOGIC);
            modeList.add(MODE_TEMPLATE_STYLE_SHEET);
            modeList.add(MODE_TEMPLATE_TEXT);
        }

        return modeList;
    }

    public static List<Integer> getLocalizeResourceList(int fetchSource) {
        List<Integer> modeList = getModeList(fetchSource);
        Iterator<Integer> iterator = modeList.iterator();
        while (iterator.hasNext()) {
            Integer next = iterator.next();
            if (LocalizeUtil.isAlreadyLocalized(next)) {
                prepareDataSourceByMode(next, fetchSource);
                iterator.remove();
            }
        }
        return modeList;
    }

    public static void goFetchLocalizeData(int fetchSource) {
        List<Integer> localizeList = getLocalizeResourceList(fetchSource);
        ResourceTracker.onResourceRequested(localizeList.size(), fetchSource);
        for (int i = 0; i < localizeList.size(); i++) {
            getLocalizeDataFromUrl(localizeList.get(i), fetchSource);
        }
    }

    public static void getLocalizeDataFromUrl(final int mode, int fetchSource) {
        if (mode == MODE_FAQ_SECTION && FaqHelper.isFaqDataAlreadyPrepared()) {
            return;
        }

        AIHelpRequest.getInstance().requestDownloadFile(mode, new ReqCallback<String>() {
            @Override
            public void onAsyncReqSuccess(String result) {
                prepareDataSourceByMode(mode, fetchSource);
                ResourceTracker.onResourceRetrieved(mode, true, fetchSource);
            }

            @Override
            public void onAsyncFailure(String url, int errorCode, String errorMsg) {
                ResourceTracker.onResourceRetrieved(mode, false, fetchSource);
                switch (mode) {
                    case MODE_FAQ_SECTION:
                        getDataAfterLocalizeFailed();
                        break;
                    case MODE_INIT_PROCESS:
                        if (TextUtils.isEmpty(url)) {
                            errorMsg = "Failed downloading custom entrance configuration, " +
                                    "please checkout your configuration in AIHelp Dashboard and re-publish.";
                        }
                        InitHelper.getInstance().onAIHelpInitializedCallback(false, errorMsg);
                        break;
                    default:
                        break;
                }
            }
        });
    }

    private static void prepareDataSourceByMode(int mode, int fetchSource) {
        switch (mode) {
            case MODE_FAQ_SECTION:
                FaqHelper.INSTANCE.prepareDataSource();
                break;
            case MODE_INIT_TEXT:
            case MODE_TEMPLATE_TEXT:
                CommonTextHelper.INSTANCE.prepareDataSource(mode);
                break;

            case MODE_INIT_LOCALE:
                LocaleStringHelper.INSTANCE.prepareDataSource();
                break;
            case MODE_INIT_PROCESS:
                ProcessEntranceHelper.INSTANCE.prepareDataSource();
                break;
            case MODE_INIT_UPLOAD_LIMIT:
                UploadLimitHelper.INSTANCE.prepareDataSource();
                break;
            case MODE_INIT_BUSINESS_LOGIC:
                InitBusinessHelper.INSTANCE.prepareDataSource();
                break;

            case MODE_TEMPLATE_STYLE_SHEET:
                StyleSheetHelper.INSTANCE.prepareDataSource();
                break;
            case MODE_TEMPLATE_BUSINESS_LOGIC:
                TemplateBusinessHelper.INSTANCE.prepareDataSource();
                break;
            case MODE_TEMPLATE_FAQ_HOT_TOPIC:
                FaqHelper.INSTANCE.prepareNotificationAndHotTopics();
                break;

        }
    }

    private static void getDataAfterLocalizeFailed() {
        AIHelpRequest.getInstance().requestGetByAsync(API.FAQ_URL, null, new ReqCallback<String>() {
            @Override
            public void onAsyncReqSuccess(String str) {
                try {
                    if (!TextUtils.isEmpty(str)) {
                        boolean success = FileUtil.writeFileToDisk(new ByteArrayInputStream((str).getBytes()),
                                LocalizeUtil.getFileLocation(MODE_FAQ_SECTION));
                        if (success) {
                            FaqHelper.INSTANCE.prepareDataSource();
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }

}
