package net.aihelp.data.localize.config;

import android.text.TextUtils;

import net.aihelp.common.API;
import net.aihelp.common.ConfigValues;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.UserProfile;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.data.localize.LocalizeHelper;
import net.aihelp.data.localize.util.LocalizeUtil;
import net.aihelp.data.model.config.IntentEntity;
import net.aihelp.data.model.config.ProcessEntity;
import net.aihelp.data.model.init.InitEntity;
import net.aihelp.data.track.statistic.StatisticTracker;
import net.aihelp.data.track.resource.tracker.TemplateResTracker;
import net.aihelp.init.InitHelper;
import net.aihelp.utils.FileUtil;
import net.aihelp.utils.TLog;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public enum ProcessEntranceHelper {

    INSTANCE;

    private final Map<String, ProcessEntity> processMap = new HashMap<>();
    private ProcessEntity currentProcess;

    private int placeholderTemplateId;
    private String placeholderCsEntrance;

    public void reset() {
        currentProcess = null;
        processMap.clear();
        TemplateResTracker.INSTANCE.reset();
    }

    public void prepareDataSource() {
        String content = FileUtil.getContentFromFile(LocalizeUtil.getFileLocation(LocalizeHelper.MODE_INIT_PROCESS));
        if (!TextUtils.isEmpty(content)) {
            JSONArray jsonArray = JsonHelper.getJsonArray(content);
            for (int i = 0; i < jsonArray.length(); i++) {
                JSONObject jsonObject = JsonHelper.getJsonObject(jsonArray, i);
                String processId = JsonHelper.optString(jsonObject, "processId");
                ProcessEntity processEntity = new ProcessEntity();
                processEntity.setEntranceId(processId);
                processEntity.setIntent(jsonObject.optInt("homeSettings"));
                processEntity.setFaqId(JsonHelper.optString(jsonObject, "faqId"));
                processEntity.setSectionId(JsonHelper.optString(jsonObject, "faqId"));
                processEntity.setVisitId(JsonHelper.optString(jsonObject, "id"));
                processEntity.setTags(JsonHelper.getJsonArray(jsonObject, "tags"));
                processEntity.setFaqEntrances(JsonHelper.getJsonArray(jsonObject, "location"));

                JSONObject intentsMenu = JsonHelper.getJsonObject(jsonObject, "intentsMenu");
                processEntity.setEnableInteraction(intentsMenu.optBoolean("alwaysShow"));
                processEntity.setSmartIntentName(JsonHelper.optString(intentsMenu, "menuText"));
                processEntity.setSmartIntentList(getBindingIntent(intentsMenu));

                boolean isDefaultEntrance = jsonObject.optInt("isDefault") == 1;
                int templateId = jsonObject.optInt("templateId");

                JSONArray textSetting = JsonHelper.getJsonArray(jsonObject, "textSetting");
                JSONObject textSettingObject = JsonHelper.getJsonObject(textSetting, 0);
                String entranceText = JsonHelper.optString(textSettingObject, "content");

                if (isDefaultEntrance) {
                    placeholderTemplateId = templateId;
                    placeholderCsEntrance = entranceText;
                }

                processEntity.setTemplateId(templateId);
                processEntity.setCsEntrance(entranceText);

                processMap.put(processId, processEntity);
            }
            InitHelper.getInstance().onAIHelpInitializedCallback(true, "");
        }

    }

    public ProcessEntity getCurrentProcess() {
        if (currentProcess == null) {
            return new ProcessEntity();
        }
        return currentProcess;
    }

    private List<IntentEntity> getBindingIntent(JSONObject jsonObject) {
        ArrayList<IntentEntity> intentList = new ArrayList<>();
        JSONArray intentsList = JsonHelper.getJsonArray(jsonObject, "intentsList");
        if (intentsList.length() > 0) {
            for (int i = 0; i < intentsList.length(); i++) {
                JSONObject intentObject = JsonHelper.getJsonObject(intentsList, i);
                IntentEntity intentEntity = new IntentEntity();
                intentEntity.setIntentId(intentObject.optInt("intentsId"));
                intentEntity.setIntentName(JsonHelper.optString(intentObject, "name"));
                intentEntity.setIntentList(getBindingIntent(intentObject));
                intentList.add(intentEntity);
            }
        }
        return intentList;
    }

    public ProcessEntity getProcess(String id) {
        ProcessEntity processEntity = processMap.get(id);
        if (processEntity == null) {
            processEntity = new ProcessEntity();
            processEntity.setEntranceId(id);
            processEntity.setTemplateId(placeholderTemplateId);
            if ("test001".equals(id)) {
                processEntity.setIntent(ConfigValues.ENTRANCE_HELP_CENTER);
            } else if ("test002".equals(id)) {
                processEntity.setIntent(ConfigValues.ENTRANCE_FAQ_SECTION);
            } else if ("test003".equals(id)) {
                processEntity.setIntent(ConfigValues.ENTRANCE_SINGLE_FAQ);
            } else if ("test004".equals(id)) {
                processEntity.setIntent(ConfigValues.ENTRANCE_CUSTOMER_SERVICE);
            } else {
                TLog.d("No related process was found, please check your process id.");
            }
        }

        // fetch template data for every entrance's first opening
        if (currentProcess == null || !currentProcess.getTemplateId().equals(processEntity.getTemplateId())) {
            fetchTemplateData(processEntity);
        }

        if (!TextUtils.isEmpty(processEntity.getCsEntrance())) {
            CustomConfig.HelpCenter.faqCSEntranceText = processEntity.getCsEntrance();
        }

        processEntity.setTrackActiveId(UUID.randomUUID().toString());
        UserProfile.ENTRANCE_TAGS = processEntity.getTags();
        currentProcess = processEntity;

        // track page view data according to entrance intent
        handleTrackLogic(processEntity);

        return processEntity;
    }

    public void fetchPlaceholderTemplateData() {
        ProcessEntity processEntity = new ProcessEntity();
        processEntity.setTemplateId(placeholderTemplateId);
        processEntity.setCsEntrance(placeholderCsEntrance);
        fetchTemplateData(processEntity);
        if (!TextUtils.isEmpty(processEntity.getCsEntrance())) {
            CustomConfig.HelpCenter.faqCSEntranceText = processEntity.getCsEntrance();
        }
    }

    private void fetchTemplateData(ProcessEntity process) {
        InitEntity initEntity = InitHelper.getInstance().getInitEntity();
        if (initEntity != null && initEntity.getSdkTemplate() != null && process != null) {
            // reset the flag of template resources to make the loading visible again
            TemplateResTracker.INSTANCE.reset();

            // reset style sheet between entrance switches
            CustomConfig.resetTemplateResources();

            // update current entrance's template url data and go fetch
            JSONObject templateObject = JsonHelper.getJsonObject(initEntity.getSdkTemplate(), process.getTemplateId());
            API.CONFIG_TEMPLATE_BUSINESS_LOGIC = JsonHelper.optString(templateObject, "business");
            API.CONFIG_TEMPLATE_STYLE_SHEET = JsonHelper.optString(templateObject, "style");
            API.CONFIG_TEMPLATE_TEXT = JsonHelper.optString(templateObject, "text");
            API.CONFIG_TEMPLATE_FAQ_HOT_TOPIC = JsonHelper.optString(templateObject, "faq");

            LocalizeHelper.goFetchLocalizeData(LocalizeHelper.FETCH_SOURCE_TEMPLATE);
        }
    }

    private void handleTrackLogic(ProcessEntity processEntity) {
        if (processEntity != null) {
            StatisticTracker tracker = StatisticTracker.getInstance();
            tracker.setupTrackHandler(processEntity.getEntranceId(), processEntity.getTrackActiveId());
            switch (processEntity.getIntent()) {
                case ConfigValues.ENTRANCE_HELP_CENTER:
                case ConfigValues.ENTRANCE_FAQ_SECTION:
                case ConfigValues.ENTRANCE_SINGLE_FAQ:
                    tracker.showAIHelp();
                    tracker.showHelpCenter();
                    break;
                case ConfigValues.ENTRANCE_CUSTOMER_SERVICE:
                    tracker.showAIHelp();
                    tracker.showCustomerService(true);
                    break;
            }
        }
    }

}
