package net.aihelp.data.logic;

import android.content.Context;
import android.text.TextUtils;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.core.mvp.AbsPresenter;
import net.aihelp.core.mvp.IRepository;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.net.mqtt.AIHelpMqtt;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.data.localize.config.ProcessEntranceHelper;
import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.data.model.rpa.step.RPAStep;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.track.statistic.StatisticTracker;
import net.aihelp.ui.cs.CustomerServiceFragment;
import net.aihelp.ui.cs.util.rpa.helper.HistoryHelper;
import net.aihelp.ui.cs.util.rpa.helper.ResponseHelper;
import net.aihelp.ui.cs.util.TicketStatusTracker;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.TLog;
import net.aihelp.utils.UploadFileHelper;

import org.json.JSONObject;

import java.util.List;

public class CustomerServicePresenter extends AbsPresenter<CustomerServiceFragment, IRepository> {

    public CustomerServicePresenter(Context context) {
        super(context, 5);
    }

    public void prepareMqtt(IMqttCallback callback, int connectionType) {
        AIHelpMqtt.getInstance().prepare(callback, connectionType);
    }

    public void requestLogin() {
        JSONObject loginParams = ResponseHelper.getLoginParams();
        EventTracker.INSTANCE.log(EventType.CS_REQUEST_LOGIN, JsonHelper.getJsonObject(loginParams.toString()));
        post(API.LOGIN, loginParams, new ReqCallback<String>() {
            @Override
            public void onReqSuccess(String result) {

                mRetryHandler.removeCallbacksAndMessages(null);

                List<Message> loginResponse = ResponseHelper.getLoginResponse(result);
                Message message = ResponseHelper.getRPAMessage(result);
                RPAStep lastRpaStep = ResponseHelper.getRPAStep(result);

                if (TicketStatusTracker.isTicketFinished) {
                    lastRpaStep = new RPAStep();
                    if (TicketStatusTracker.isTicketWaitForAskingResolveStatus || TicketStatusTracker.isTicketWaitForRating) {
                        lastRpaStep.setNextStep(RPAStep.STEP_EVALUATE_SERVICE);
                    } else {
                        lastRpaStep.setNextStep(RPAStep.STEP_NEW_CONVERSATION);
                    }
                } else if (TicketStatusTracker.isTicketActive && TicketStatusTracker.isTicketServingByAgent()) {
                    lastRpaStep.setNextStep(RPAStep.STEP_MANUAL_INPUT);
                } else if (TicketStatusTracker.isTicketRejected) {
                    lastRpaStep.setNextStep(RPAStep.STEP_NEW_CONVERSATION);
                }

                mView.onLogin(loginResponse, message, lastRpaStep);
            }

            @Override
            public void onFailure(String url, int errorCode, String errorMsg) {
                mRetryHandler.handleRetryRequest(errorCode, errorMsg);
            }
        });
    }

    public void logout() {
        post(API.LOGOUT, null, null);
        // logout 重置状态，会导致返回到帮助中心的时候不再有消息入口，因为逻辑上没有进行中客诉了
        // TicketStatusTracker.resetTicketStatusFlags();
    }

    public void chatWithSupport(long timestamp, JSONObject chatParams) {
        try {
            EventTracker.INSTANCE.log(EventType.CS_MESSAGE_SENT, chatParams);
            StatisticTracker.getInstance().setWhenMessageSent();
            post(API.SEND_MESSAGE, chatParams, new ReqCallback<String>() {
                @Override
                public void onReqSuccess(String result) {
                    ResponseHelper.notifyMqttPush(result);

                    Message message = ResponseHelper.getRPAMessage(result);
                    RPAStep rpaStep = ResponseHelper.getRPAStep(result);

                    // use (the response message timestamp - 1) to update user message's timestamp, to ensure
                    // the list render order and the startTime in {@link #goBackToPreviousStep()} can
                    // overlap user message so that we could remove the user message after going back to previous step
                    mView.updateMessageStatus(true, timestamp,
                            message.getTimestamp() - 1);

                    if (rpaStep.isEnableUpload()) {
                        UploadFileHelper.INSTANCE.tryUploadLog(true);
                    } else {
                        if (message.isNormalMessage()) {
                            mView.updateChatList(message);
                        }
                        if (rpaStep.getNextStep() != RPAStep.STEP_IGNORE_THIS) {
                            mView.updateBottomLayout(message, rpaStep);
                        }
                    }

                    // try to reconnect mqtt after every user action when mqtt is disconnected
                    if (!MqttConfig.getInstance().isConnected()) {
                        ApiExecutorFactory.getHandlerExecutor().runAsyncDelayed(new Runnable() {
                            @Override
                            public void run() {
                                mView.prepareMqtt(MqttConfig.TYPE_CONNECTION_RECONNECT);
                            }
                        }, 2000);
                    }
                }

                @Override
                public void onFailure(String url, int errorCode, String errorMsg) {
                    EventTracker.INSTANCE.log(EventType.CS_MESSAGE_SEND_FAILED);
                    mView.updateMessageStatus(false, timestamp, timestamp);
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void goBackToPreviousStep() {
        EventTracker.INSTANCE.log(EventType.CS_GO_BACK_TO_PREVIOUS_STEP);
        get(API.PREVIOUS_STEP, null, new ReqCallback<String>() {
            @Override
            public void onReqSuccess(String result) {

                cleanMsgListAfterGoBackToPreviousStep(result);

                Message message = ResponseHelper.getRPAMessage(result);
                RPAStep rpaStep = ResponseHelper.getRPAStep(result);

                if (rpaStep.isEnableUpload()) {
                    UploadFileHelper.INSTANCE.tryUploadLog(true);
                } else {
                    if (rpaStep.getNextStep() != RPAStep.STEP_IGNORE_THIS) {
                        mView.updateBottomLayout(message, rpaStep);
                    }
                }
            }
        });
    }

    private void cleanMsgListAfterGoBackToPreviousStep(String prevStepResponse) {
        try {
            JSONObject jsonObject = new JSONObject(prevStepResponse);
            long startTime = jsonObject.optLong("startTime");
            long endTime = jsonObject.optLong("endTime");
            mView.notifyMessageWithdrawn(startTime, endTime);
        } catch (Exception e) {
            // ignored
        }
    }

    public void getLastConversation() {
        EventTracker.INSTANCE.log(EventType.CS_GET_LAST_CONVERSATION);
        if (!AppInfoUtil.validateNetwork(mContext)) {
            mView.onLastConversationRetrieved(null);
            return;
        }
        try {
            JSONObject hashMap = new JSONObject();
            hashMap.put("userId", UserProfile.USER_ID);
            hashMap.put("processId", ProcessEntranceHelper.INSTANCE.getCurrentProcess().getEntranceId());
//            hashMap.put("deviceId", DeviceUuidFactory.id(mContext));
            hashMap.put("messageTime", TicketStatusTracker.firstMessageTimeStampInList);
            hashMap.put("language", Const.ORIGINAL_LANGUAGE);
            hashMap.put("pageSize", 20);
            get(API.GET_LAST_CONVERSATION, hashMap, new ReqCallback<String>() {
                @Override
                public void onReqSuccess(String result) {
                    List<Message> list = HistoryHelper.getHistoryList(result, HistoryHelper.HISTORY_TYPE_REFRESH);
                    TicketStatusTracker.setFirstMessageTimeStampInList(list);
                    mView.onLastConversationRetrieved(list);
                }

                @Override
                public void onFailure(String url, int errorCode, String errorMsg) {
                    mView.onLastConversationRetrieved(null);
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void refreshUnreadMessageCount() {
        try {
            TicketStatusTracker.hasUnreadMsg = false;
            JSONObject params = new JSONObject();
            params.put("appid", Const.APP_ID);
            params.put("uid", UserProfile.USER_ID);
            get(API.FETCH_NEW_MSG, params, new ReqCallback<String>() {
                @Override
                public void onReqSuccess(String result) {
                    try {
                        if (!TextUtils.isEmpty(result)) {
                            JSONObject jsonObject = new JSONObject(result);
                            Const.TOGGLE_FETCH_MESSAGE = jsonObject.optBoolean("isHaveChat");
                            int newCount = jsonObject.optInt("cs_message_count");
                            mSp.put(UserProfile.USER_ID, Math.max(0, newCount));

                            // Give caller a unread message count of 0 to notify them user has read
                            // all messages after cached latest message count.
                            // changed at 2021/08/11 since v2.5.1
                            if (Const.TOGGLE_OPEN_UNREAD_MSG && Const.sMessageListener != null) {
                                Const.sMessageListener.onMessageCountArrived(0);
                            }
                        }
                    } catch (Exception e) {
                        TLog.d("Conversation fetch new msg failed, because " + e.toString());
                    }
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void updateCachedUnreadMessageCount(boolean isAgentMessage, boolean withdrawMessage) {
        if (isAgentMessage) {
            int cacheCount = mSp.getInt(UserProfile.USER_ID, 0);
            mSp.put(UserProfile.USER_ID, withdrawMessage ? --cacheCount : ++cacheCount);
        }
    }

    @Override
    public void onRetryRequest() {
        requestLogin();
    }

}
