package net.aihelp.data.logic;

import android.content.Context;
import android.text.TextUtils;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.core.mvp.AbsPresenter;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.mqtt.AIHelpMqtt;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.concurrent.ApiExecutor;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;

import net.aihelp.data.event.TaskCenterUnreadEvent;
import net.aihelp.data.local.FaqRepository;
import net.aihelp.data.localize.data.FaqHelper;
import net.aihelp.data.localize.util.LocalizeUtil;
import net.aihelp.data.model.faq.FaqListEntity;
import net.aihelp.data.model.faq.FaqContentEntity;
import net.aihelp.ui.faq.BaseFaqFragment;
import net.aihelp.ui.faq.FaqContentFragment;
import net.aihelp.ui.faq.FaqListFragment;
import net.aihelp.ui.faq.FaqHomeFragment;
import net.aihelp.utils.FileUtil;
import net.aihelp.data.localize.LocalizeHelper;
import net.aihelp.utils.Styles;
import net.aihelp.utils.TLog;

import org.json.JSONObject;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

public class FaqPresenter extends AbsPresenter<BaseFaqFragment<?>, FaqRepository> {

    private static final ApiExecutor sApiExecutor = ApiExecutorFactory.getHandlerExecutor();

    public FaqPresenter(Context context) {
        super(context);
    }

    private void fetchFaqDataSourceOnDemand(final String sectionOrFaqId) {
        if (isNetworkAvailable()) {
            mView.showLoading();
            String faqUrl = LocalizeUtil.getUrl(LocalizeHelper.MODE_FAQ_SECTION);
            if (Pattern.compile(".+\\.(json)$").matcher(faqUrl).matches()) {
                AIHelpRequest.getInstance().requestDownloadFile(LocalizeHelper.MODE_FAQ_SECTION, new ReqCallback<String>() {
                    @Override
                    public void onAsyncReqSuccess(String result) {
                        // 下载完成后通知准备数据
                        FaqHelper.INSTANCE.prepareDataSource(new Runnable() {
                            @Override
                            public void run() {
                                refreshFaqs(sectionOrFaqId);
                            }
                        });
                    }

                    @Override
                    public void onFailure(String url, int errorCode, String errorMsg) {
                        getFaqFromApiAfterLocalizeFailed(sectionOrFaqId);
                    }
                });
            }
        } else {
            mView.showNetError();
        }
    }

    private void getFaqFromApiAfterLocalizeFailed(final String sectionOrFaqId) {
        get(API.FAQ_URL, null, new ReqCallback<String>() {
            @Override
            public void onAsyncReqSuccess(String str) {
                try {
                    if (!TextUtils.isEmpty(str)) {
                        boolean success = FileUtil.writeFileToDisk(new ByteArrayInputStream((str).getBytes()),
                                LocalizeUtil.getFileLocation(LocalizeHelper.MODE_FAQ_SECTION));
                        if (success) {
                            FaqHelper.INSTANCE.prepareDataSource(new Runnable() {
                                @Override
                                public void run() {
                                    refreshFaqs(sectionOrFaqId);
                                }
                            });
                        } else {
                            mView.showEmpty();
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }

    private void refreshFaqs(String sectionOrFaqId) {
        if (mView != null && !mView.isDetached()) {
            mView.restoreViewState();
            if (mView instanceof FaqHomeFragment || mView instanceof FaqListFragment) {
                goFetchFAQDataSource(sectionOrFaqId);
            }

            if (mView instanceof FaqContentFragment) {
                goFetchQuestionContent(null, sectionOrFaqId, "");
            }
        }
    }

    public void goFetchFAQDataSource(String sectionId) {
        if (FaqHelper.isFaqDataAlreadyPrepared()) {
            List<FaqListEntity> dataSource;
            boolean rootPage = TextUtils.isEmpty(sectionId);
            if (rootPage) {
                // 如果没有 sectionId，那就是顶级分类
                dataSource = FaqHelper.INSTANCE.getRootSections();
            } else if (mRepo.checkWhetherHasSubSection(sectionId)) {
                // 如果 sectionId 有值，那就看看这个 sectionId 下边有没有子分类
                // 要是有子分类，就从 map 中直接取出来对应的子分类列表
                dataSource = FaqHelper.INSTANCE.getSubSections(sectionId);
            } else {
                // 如果当前 sectionId 下也没有子分类，那就是普通分类，直接获取 faq 问题列表就可以了
                dataSource = FaqHelper.INSTANCE.getQuestionList(sectionId);
            }
            sApiExecutor.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (mView != null && !mView.isDetached()) {
                        mView.refreshList(dataSource, sectionId);
                    }
                }
            });
        } else {
            fetchFaqDataSourceOnDemand(sectionId);
        }
    }

    public void goFetchQuestionContent(String sectionId, String faqId, String searchTerm) {
        if (FaqHelper.isFaqDataAlreadyPrepared()) {
            FaqContentEntity faq = FaqHelper.INSTANCE.getFaqById(sectionId, faqId);
            sApiExecutor.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (faq != null && !TextUtils.isEmpty(faq.getFaqContent())) {
                        final FaqContentEntity highlightedFaq = Styles.getFAQWithHighlightedSearchTerms(mContext, faq, searchTerm);
                        if (mView == null || mView.isDetached()) {
                            return;
                        }
                        mView.refreshQuestionContent(highlightedFaq != null ? highlightedFaq : faq, searchTerm);
                    } else {
                        mView.showEmpty();
                    }
                }
            });
        } else {
            fetchFaqDataSourceOnDemand(faqId);
        }
    }

    public void goQueryFAQList(final String query) {
        sApiExecutor.runAsync(new Runnable() {
            @Override
            public void run() {
                final ArrayList<FaqListEntity> matchedFaqList;
                if (TextUtils.isEmpty(query) || FaqHelper.INSTANCE.getRawFlatFaqArray().length() == 0) {
                    matchedFaqList = null;
                } else {
                    matchedFaqList = mRepo.getMatchedFaqList(query);
                }
                sApiExecutor.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        mView.refreshList(matchedFaqList);
                    }
                });
            }
        });
    }

    public void prepareFAQNotification() {
        if (Const.TOGGLE_OPEN_FAQ_NOTIFICATION) {
            IMqttCallback mqttCallback = MqttCallbackImpl.getInstance();
            mqttCallback.updateHostView(mView);
            AIHelpMqtt.getInstance().prepare(MqttConfig.TYPE_FAQ, mqttCallback);
        }

        try {
            JSONObject params = new JSONObject();
            params.put("appid", Const.APP_ID);
            params.put("uid", UserProfile.USER_ID);
            get(API.FETCH_NEW_MSG, params, new ReqCallback<String>() {
                @Override
                public void onReqSuccess(String result) {
                    try {
                        if (!TextUtils.isEmpty(result)) {
                            JSONObject jsonObject = new JSONObject(result);
                            int newCount = jsonObject.optInt("cs_message_count");
                            boolean isTicketActive = jsonObject.optBoolean("isHaveChat");
                            int cachedCount = mSp.getInt(UserProfile.USER_ID, 0);
                            if (Math.max(0, newCount - cachedCount) > 0) {
                                mView.showEntranceWithNotification(true, false);
                            } else if (isTicketActive) {
                                mView.showEntranceWithNotification(false, false);
                            }
                        }
                    } catch (Exception e) {
                        TLog.d("FAQ fetch new msg failed, because " + e.toString());
                    }
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }

        AIHelpRequest.getInstance().requestPostByJson(API.TICKET_UNREAD_STATUS, new JSONObject(), new ReqCallback<String>() {
            @Override
            public void onReqSuccess(String result) {
                EventBus.getDefault().post(new TaskCenterUnreadEvent(Boolean.valueOf(result)));
            }
        });

    }

    public boolean hasSubSection(String sectionId) {
        return mRepo.checkWhetherHasSubSection(sectionId);
    }

    public boolean shouldShowQuestionFooter(String faqId, long lastUpdateTime) {
        return mRepo.shouldShowQuestionFooter(faqId, lastUpdateTime);
    }

}
