package net.aihelp.data.logic;

import net.aihelp.BuildConfig;
import net.aihelp.common.API;
import net.aihelp.common.SpKeys;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.model.rpa.step.RPAStep;
import net.aihelp.ui.cs.CustomerServiceFragment;
import net.aihelp.ui.cs.util.TicketStatusTracker;
import net.aihelp.ui.cs.util.rpa.helper.ResponseHelper;
import net.aihelp.ui.helper.BreakReleaseHelper;
import net.aihelp.utils.ListUtil;
import net.aihelp.utils.SpUtil;

import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import androidx.fragment.app.Fragment;

public enum MessagePoller {

    INSTANCE;

    public static final int POLL_SOURCE_LOGIN = 1;
    public static final int POLL_SOURCE_FAILURE = 2;
    public static final int POLL_SOURCE_RECONNECT = 3;
    public static final int POLL_SOURCE_SCHEDULER = 4;

    private ScheduledFuture<?> newMessageFetcher;
    private WeakReference<CustomerServiceFragment> csFragment;

    public void startMessagePolling(Fragment fragment) {
        if (fragment instanceof CustomerServiceFragment) {
            csFragment = new WeakReference<>((CustomerServiceFragment) fragment);
        }
        if (BreakReleaseHelper.isBreak()) {
            TicketStatusTracker.ticketPollInterval = 10;
        }
        if (newMessageFetcher == null) {
            newMessageFetcher = Executors.newScheduledThreadPool(1).scheduleAtFixedRate(new Runnable() {
                @Override
                public void run() {
                    if (MqttConfig.getInstance().isConnected()) {
                        return;
                    }
                    fetchMessagesSinceLatest(POLL_SOURCE_SCHEDULER);
                }
            }, TicketStatusTracker.ticketPollInterval, TicketStatusTracker.ticketPollInterval, TimeUnit.SECONDS);
        }
    }

    public void stopMessagePolling() {
        if (newMessageFetcher != null) {
            newMessageFetcher.cancel(true);
            newMessageFetcher = null;
        }
    }

    public void fetchMessagesSinceLatest(int pollSource) {
        try {
            if (csFragment != null && csFragment.get() != null) {
                CustomerServiceFragment customerServiceFragment = csFragment.get();
                JSONObject params = new JSONObject();
                params.put("messageTime", customerServiceFragment.getLastAgentMessageTimestamp());
                params.put("pollSource", pollSource);
                AIHelpRequest.getInstance().requestGetByAsync(API.POLL_MESSAGE_AFTER_MQTT_DISCONNECT, params, new ReqCallback<String>() {
                    @Override
                    public void onReqSuccess(String result) {
                        List<Message> pollResponse = ResponseHelper.getPollResponse(result);
                        Message message = ResponseHelper.getRPAMessage(result);
                        RPAStep lastRpaStep = ResponseHelper.getRPAStep(result);
                        if (TicketStatusTracker.isTicketFinished) {
                            lastRpaStep = new RPAStep();
                            if (TicketStatusTracker.isTicketWaitForAskingResolveStatus || TicketStatusTracker.isTicketWaitForRating) {
                                lastRpaStep.setNextStep(RPAStep.STEP_EVALUATE_SERVICE);
                            } else {
                                lastRpaStep.setNextStep(RPAStep.STEP_NEW_CONVERSATION);
                            }
                        } else if (TicketStatusTracker.isTicketActive && TicketStatusTracker.isTicketServingByAgent()) {
                            lastRpaStep.setNextStep(RPAStep.STEP_MANUAL_INPUT);
                        } else if (TicketStatusTracker.isTicketRejected) {
                            lastRpaStep.setNextStep(RPAStep.STEP_NEW_CONVERSATION);
                        }
                        if (!ListUtil.isListEmpty(pollResponse)) {
                            customerServiceFragment.updateChatList(pollResponse, false);
                        }
                        if (!TicketStatusTracker.isTicketServingByAnswerBot()) {
                            customerServiceFragment.updateBottomLayout(message, lastRpaStep, false);
                        }
                    }
                });
            }
        } catch (Exception e) {
            // ignored
        }
    }

}
