package net.aihelp.data.logic;

import android.os.Handler;
import android.os.Message;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.config.AIHelpContext;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.ui.helper.BreakReleaseHelper;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.utils.SpUtil;
import net.aihelp.utils.TLog;

import org.json.JSONObject;

import androidx.annotation.NonNull;

public class UnreadMessagePoller extends Handler {

    private final static int POLL_UNREAD_MESSAGE = 100;

    public UnreadMessagePoller() {
        super();
    }

    public void start() {
        stop();
        sendEmptyMessage(POLL_UNREAD_MESSAGE);
    }

    public void stop() {
        removeCallbacksAndMessages(null);
    }

    @Override
    public void handleMessage(@NonNull Message msg) {
        if (isPollingEnable()) {
            fetchUnreadMessageCount();
        }
        sendEmptyMessageDelayed(POLL_UNREAD_MESSAGE, getPollingLimit());
    }

    private void fetchUnreadMessageCount() {
        if (interceptFetchRequest()) return;
        JSONObject params = JsonHelper.getJsonObject();
        JsonHelper.put(params, "appid", Const.APP_ID);
        JsonHelper.put(params, "uid", UserProfile.USER_ID);
        AIHelpRequest.getInstance().requestGetByAsync(API.FETCH_NEW_MSG, params, new ReqCallback<String>() {
            @Override
            public void onReqSuccess(String result) {
                JSONObject jsonObject = JsonHelper.getJsonObject(result);
                Const.TOGGLE_FETCH_MESSAGE = jsonObject.optBoolean("isHaveChat");
                int newCount = jsonObject.optInt("cs_message_count");
                int cachedCount = Math.max(0, SpUtil.getInstance().getInt(UserProfile.USER_ID));
                if (Const.sMessageListener != null) {
                    Const.sMessageListener.onMessageCountArrived(Math.max(0, newCount - cachedCount));
                }
            }
        });
    }

    private boolean interceptFetchRequest() {
        if (Const.IS_SDK_SHOWING) {
            TLog.d("AIHelp", "AIHelp session is visible to user, do not need fetch for unread messages.");
            return true;
        } else if (!Const.TOGGLE_FETCH_MESSAGE) {
            TLog.d("AIHelp", String.format("Current user(%s) does not have any active tickets at present.", UserProfile.USER_ID));
            return true;
        } else if (UserProfile.USER_ID.equals(DeviceUuidFactory.id(AIHelpContext.getInstance().getContext()))) {
            TLog.d("AIHelp", "The userId you're using for unread message polling is AIHelp's generated deviceId, " +
                    "please verify if this is what you want.");
        }
        return false;
    }

    private long getPollingLimit() {
        if (BreakReleaseHelper.isBreak()) {
            return 10 * 1000L;
        } else if (!isPollingEnable()) {
            return 300 * 1000L;
        } else {
            return Const.LIMIT_CHECKING_UNREAD * 1000L;
        }
    }

    private boolean isPollingEnable() {
        return Const.TOGGLE_OPEN_UNREAD_MSG && Const.LIMIT_CHECKING_UNREAD > 0;
    }

}
