package net.aihelp.init;

import android.content.Context;

import net.aihelp.BuildConfig;
import net.aihelp.common.Const;
import net.aihelp.config.AIHelpContext;
import net.aihelp.config.ApiConfig;
import net.aihelp.config.UserConfig;
import net.aihelp.config.enums.PublishCountryOrRegion;
import net.aihelp.config.enums.PushPlatform;
import net.aihelp.config.enums.ShowConversationMoment;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.exception.AIHelpInitException;
import net.aihelp.ui.listener.OnAIHelpInitializedCallback;
import net.aihelp.ui.listener.OnAIHelpSessionCloseCallback;
import net.aihelp.ui.listener.OnAIHelpSessionOpenCallback;
import net.aihelp.ui.listener.OnSpecificUrlClickedCallback;
import net.aihelp.ui.listener.OnMessageCountArrivedCallback;
import net.aihelp.ui.listener.OnNetworkCheckResultCallback;
import net.aihelp.ui.listener.OnSpecificFormSubmittedCallback;
import net.aihelp.utils.DeviceUuidFactory;

public class AIHelpSupport {

    /**
     * Initialize AIHelp support
     * <p>
     * When initializing AIHelp you must at least pass four tokens.
     * You should initialize AIHelp at the beginning of your app lifecycle, ideally at the top of Application#onCreate.
     * This method can throw the InstallException asynchronously if the install keys are not in the correct format.
     *
     * @param context This should be your application instance
     * @param appKey  This is your developer API Key
     * @param domain  This is your domain name without any http:// or forward slashes
     * @param appId   This is the unique ID assigned to your app
     */
    public static void init(Context context, String appKey, String domain, String appId) {
        init(context, appKey, domain, appId, Const.CORRECT_LANGUAGE);
    }

    /**
     * Initialize AIHelp support
     * <p>
     * When initializing AIHelp you must at least pass four tokens.
     * You should initialize AIHelp at the beginning of your app lifecycle, ideally at the top of Application#onCreate.
     * This method can throw the InstallException asynchronously if the install keys are not in the correct format.
     *
     * @param context  This should be your application instance
     * @param appKey   This is your developer API Key
     * @param domain   This is your domain name without any http:// or forward slashes
     * @param appId    This is the unique ID assigned to your app
     * @param language This is your expected init language
     */
    public static void init(final Context context, final String appKey, final String domain,
                            final String appId, final String language) throws AIHelpInitException {
        AIHelpCore.getInstance().init(context, appKey, domain, appId, language);
    }

    /**
     * Register callback for the process of AIHelp's initialization.
     * <p>
     * After you register this callback, SDK will let you know if the init work is done.
     * You can call this method either before or after the {@link #init(Context, String, String, String)} call.
     *
     * @param listener callback for AIHelp initialization
     */
    public static void setOnAIHelpInitializedCallback(OnAIHelpInitializedCallback listener) {
        AIHelpCore.getInstance().setOnAIHelpInitializedCallback(listener);
    }

    public static boolean show(String entranceId) {
        return AIHelpCore.getInstance().show(new ApiConfig.Builder().setEntranceId(entranceId).build());
    }

    public static boolean show(ApiConfig config) {
        return AIHelpCore.getInstance().show(config);
    }

    public static void showSingleFAQ(String faqId, ShowConversationMoment moment) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().showSingleFAQ(faqId, moment);
            }
        });
    }

    public static void showUrl(final String url) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().showUrl(url);
            }
        });
    }

    /**
     * Update a user's profile via UserConfig.
     * <p>
     * Please check {@link UserConfig} for more detail information.
     *
     * @param config configs which contains all information about a user.
     * @see UserConfig
     */
    public static void updateUserInfo(final UserConfig config) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().updateUserInfo(config);
            }
        });
    }

    /**
     * Clear the values set to a user, reset the userId to deviceId, userName to 'anonymous'.
     */
    public static void resetUserInfo() {
        Context context = AIHelpContext.getInstance().getContext();
        if (context != null) {
            UserConfig resetUser = new UserConfig.Builder()
                    .setUserId(DeviceUuidFactory.id(context))
                    .setServerId("-1")
                    .setUserName("anonymous")
                    .build();
            updateUserInfo(resetUser);
        }
    }

    /**
     * Change the SDK language. By default, the device's prefered language is used.
     * The call will fail in the following cases :
     * 1. If a AIHelp session is already active at the time of invocation
     * 2. Language code is incorrect
     * 3. Corresponding localization file is not found
     *
     * @param sdkLanguage the string representing the language code. For example, use 'fr' for French.
     */
    public static void updateSDKLanguage(String sdkLanguage) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().updateSDKLanguage(sdkLanguage);
            }
        });
    }

    /**
     * start in-app unread message count polling
     * <p>
     * This is a schedule work, get unread message count every five minutes.
     *
     * @param listener callback for unread message polling
     */
    public static void startUnreadMessageCountPolling(OnMessageCountArrivedCallback listener) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().startUnreadMessageCountPolling(listener);
            }
        });
    }

    /**
     * stop the in-app unread message count polling
     */
    public static void stopUnreadMessageCountPolling() {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().stopUnreadMessageCountPolling();
            }
        });
    }

    /**
     * Set log path for uploading.
     * <p>
     * In order to serve your customers well, you can upload customer-related-logs when tickets are created or
     * specific forms are submitted.
     *
     * @param logPath the absolute path of log, which will be uploaded when needed
     */
    public static void setUploadLogPath(String logPath) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setUploadLogPath(logPath);
            }
        });
    }

    /**
     * @param pushToken    the pushToken received from the push notification servers.
     * @param pushPlatform the specific push platform, must be of
     *                     {@link PushPlatform#APNS}, {@link PushPlatform#FIREBASE},
     *                     {@link PushPlatform#JPUSH}, {@link PushPlatform#GETUI}
     *                     <p>
     *                     set the pushToken and platform to enable push notifications.
     *                     NOTE: You must get the specific push sdk in your App,
     *                     and call {@link #updateUserInfo(UserConfig)} BEFORE this invocation.
     */
    public static void setPushTokenAndPlatform(String pushToken, PushPlatform pushPlatform) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().setPushTokenAndPlatform(pushToken, pushPlatform);
            }
        });
    }

    /**
     * Set up host address for network check, without result callback.
     * Check {@link #setNetworkCheckHostAddress(String, OnNetworkCheckResultCallback)} for more information.
     *
     * @param hostAddress host address for network checking, without schemes such 'https://' or 'http://'.
     *                    For example, you can pass in 'www.google.com' or just 'google.com', no schemes are needed.
     */
    public static void setNetworkCheckHostAddress(String hostAddress) {
        setNetworkCheckHostAddress(hostAddress, null);
    }

    /**
     * Set up host address for network check with result callback.
     * <p>
     * With this api, you can get the network check result passing back to you.
     *
     * @param hostAddress host address for network checking, without schemes such 'https://' or 'http://'.
     *                    For example, you can pass in 'www.google.com' or just 'google.com', no schemes are needed.
     * @param listener    network check result callback, you can get the check result via this listener
     */
    public static void setNetworkCheckHostAddress(String hostAddress, OnNetworkCheckResultCallback listener) {
        AIHelpCore.getInstance().setNetworkCheckHostAddress(hostAddress, listener);
    }

    /**
     * Register callback for the submit of particular forms.
     * <p>
     * With this api, you can get callback when the specific forms you configured in AIHelp dashboard are submitted.
     *
     * @param callback callback for form submitted
     */
    public static void setOnSpecificFormSubmittedCallback(OnSpecificFormSubmittedCallback callback) {
        AIHelpCore.getInstance().setOnSpecificFormSubmittedCallback(callback);
    }

    /**
     * Register callback for AIHelp's session visibility
     *
     * @param callback callback when AIHelp session is visible
     */
    public static void setOnAIHelpSessionOpenCallback(OnAIHelpSessionOpenCallback callback) {
        AIHelpCore.getInstance().setOnAIHelpSessionOpenCallback(callback);
    }

    /**
     * Register callback for AIHelp's session visibility
     *
     * @param callback callback when AIHelp session is closed by users
     */
    public static void setOnAIHelpSessionCloseCallback(OnAIHelpSessionCloseCallback callback) {
        AIHelpCore.getInstance().setOnAIHelpSessionCloseCallback(callback);
    }

    /**
     * Register callback for specific url clicked by users.
     *
     * @param callback Whenever a url contains 'js-bridge=enable' is clicked
     */
    public static void setOnSpecificUrlClickedCallback(OnSpecificUrlClickedCallback callback) {
        AIHelpCore.getInstance().setOnSpecificUrlClickedCallback(callback);
    }

    /**
     * AIHelp provide additional support for some country or regions.
     *
     * @param countryOrRegion ISO country code, please check https://www.iso.org/iso-3166-country-codes.html to learn more.
     */
    public static void additionalSupportFor(PublishCountryOrRegion countryOrRegion) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpCore.getInstance().additionalSupportFor(countryOrRegion);
            }
        });
    }

    /**
     * Get current AIHelp SDK version
     *
     * @return AIHelp SDK version
     */
    public static String getSDKVersion() {
        return BuildConfig.SDK_VERSION;
    }

    /**
     * Whether AIHelp session is visible to users
     *
     * @return whether sdk is active
     */
    public static boolean isAIHelpShowing() {
        return Const.IS_SDK_SHOWING;
    }

    /**
     * Enable develop logs of AIHelp
     */
    public static void enableLogging(boolean enable) {
        AIHelpCore.getInstance().enableLogging(enable);
    }

    /**
     * Finish all AIHelp sessions
     */
    public static void close() {
        AIHelpCore.getInstance().close();
    }

}
