package net.aihelp.init;

import android.content.Context;
import android.text.TextUtils;
import android.util.Log;

import net.aihelp.BuildConfig;
import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.SpKeys;
import net.aihelp.config.AIHelpContext;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.data.localize.LocalizeHelper;
import net.aihelp.data.logic.RequestRetryHandler;
import net.aihelp.data.model.init.InitEntity;
import net.aihelp.data.track.resource.ResourceTracker;
import net.aihelp.ui.helper.MessageSyncHelper;
import net.aihelp.utils.DateFormatUtil;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.utils.SpUtil;

import org.json.JSONObject;

public class InitHelper implements RequestRetryHandler.OnRetryRequestListener {

    public static final int ERROR_INVALID_APP_ID = 4100;

    private static InitHelper sInstance;
    private final RequestRetryHandler mRetryHandler;
    private boolean requireCallback;
    private InitEntity initEntity;

    private InitHelper() {
        this.mRetryHandler = new RequestRetryHandler(this, 5);
    }

    public static InitHelper getInstance() {
        if (sInstance == null) {
            sInstance = new InitHelper();
        }
        return sInstance;
    }

    public InitEntity getInitEntity() {
        return initEntity;
    }

    // AIHelpSupport APIs are all running on UI thread, so if there are cached init responses,
    // the initialization will be succeed directly, but because of the main looper message
    // queue, the init callback haven't been set.
    // Transfer init task to another message queue and delayed for 0.5s to make sure init callback can work well.
    public void runInitRelatedTask(Context context) {
        goLogDauStatus(context);
        requestInit(true);
        MessageSyncHelper.syncLogMessage();
    }

    public void onAIHelpInitializedCallback(boolean isSuccess, String message) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                AIHelpContext.successfullyInit.set(isSuccess);
                if (requireCallback && Const.sInitializedListener != null) {
                    Const.sInitializedListener.onAIHelpInitialized(isSuccess, message);
                }
            }
        });
    }

    private void goLogDauStatus(Context context) {
        long initTime = SpUtil.getInstance().getLong(SpKeys.LOG_DAU_TIME);
        if (!DateFormatUtil.isToday(initTime)) {
            try {
                JSONObject dauParams = new JSONObject();
                // DO NOT try to change this deviceId EVER, or the DAU data will be messed, which will
                // results in pricing-related issues.
                dauParams.put("deviceid", DeviceUuidFactory.id(context));
                AIHelpRequest.getInstance().requestPostByJson(API.LOG_DAU_URL, dauParams, new ReqCallback<String>() {
                    @Override
                    public void onReqSuccess(String result) {
                        SpUtil.getInstance().put(SpKeys.LOG_DAU_TIME, System.currentTimeMillis());
                    }
                });
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public void requestInit(boolean requireCallback) {
        this.requireCallback = requireCallback;

        String spPrefix = String.format("%s_%s_%s", Const.APP_ID, Const.ORIGINAL_LANGUAGE, BuildConfig.SDK_VERSION);
        final long lastInitTime = SpUtil.getInstance().getLong(spPrefix + SpKeys.LAST_INIT_TIME, System.currentTimeMillis());
        int requestLimit = SpUtil.getInstance().getInt(spPrefix + SpKeys.INIT_REQUEST_LIMIT);
        final String cachedInitData = SpUtil.getInstance().getString(spPrefix + SpKeys.CACHE_INIT_RESPONSE);

        // reset localize data to ensure data's accuracy
        LocalizeHelper.resetLocalizeData();

        if (TextUtils.isEmpty(cachedInitData) || requestLimit < 0 || System.currentTimeMillis() - lastInitTime > requestLimit) {
            ResourceTracker.markInitStarted();
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("domain", API.HOST_URL);

                AIHelpRequest.getInstance().requestGetByAsync(API.INIT_URL, jsonObject, new ReqCallback<String>() {

                    @Override
                    public void onAsyncReqSuccess(String result) {
                        if (mRetryHandler != null) {
                            mRetryHandler.removeCallbacksAndMessages(null);
                        }
                        cacheInitResponse(result, System.currentTimeMillis());
                    }

                    @Override
                    public void onAsyncFailure(String url, int errorCode, String errorMsg) {
                        switch (errorCode) {
                            case ERROR_INVALID_APP_ID:
                                onAIHelpInitializedCallback(false, "Invalid init parameters");
                                break;
                            default:
                                ResourceTracker.markInitEnded(false);
                                if (TextUtils.isEmpty(cachedInitData)) {
                                    if (mRetryHandler != null) {
                                        mRetryHandler.handleRetryRequest(errorCode, errorMsg);
                                    }
                                } else {
                                    cacheInitResponse(cachedInitData, lastInitTime);
                                }
                                break;
                        }
                    }
                });
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else {
            cacheInitResponse(cachedInitData, lastInitTime);
        }
    }

    private void cacheInitResponse(String initResponse, long lastInitTime) {
        initEntity = JsonHelper.toJavaObject(initResponse, InitEntity.class);
        if (initEntity != null) {
            int cachePeriod = initEntity.getRequestLimit();
            String spPrefix = String.format("%s_%s_%s", Const.APP_ID, Const.ORIGINAL_LANGUAGE, BuildConfig.SDK_VERSION);
            SpUtil.getInstance().put(spPrefix + SpKeys.INIT_REQUEST_LIMIT, cachePeriod < 0 ? cachePeriod : cachePeriod * 60 * 1000);
            SpUtil.getInstance().put(spPrefix + SpKeys.CACHE_INIT_RESPONSE, initResponse);
            SpUtil.getInstance().put(spPrefix + SpKeys.LAST_INIT_TIME, lastInitTime);

            API.CDN_URL = initEntity.getCdnUrl();
            API.FAQ_URL = initEntity.getFaqdata();
            API.OP_URL = initEntity.getFaqYYdata();
            API.UPLOAD_IMAGE_URL = initEntity.getUpload();
            API.UPLOAD_VIDEO_URL = initEntity.getUploadVideo();
            API.UPLOAD_LOG_URL = initEntity.getUploadLog();
            API.UPLOAD_ATTACHMENT_URL = initEntity.getUploadFile();
            API.M_FAQ_URL = initEntity.getShowfaq();

            API.LOG_COUNT_URL = initEntity.getPoint();
            API.CONFIG_INIT_LOCALE_FILE = initEntity.getLocaleFile();
            API.CONFIG_INIT_UPLOAD_LIMIT = initEntity.getSdkCustomUpload();

            API.MQTT_IP = initEntity.getSvrip();
            API.MQTT_PORT = initEntity.getSvrport();
            API.MQTT_TOPIC = initEntity.getTopic();
            if (!TextUtils.isEmpty(initEntity.getPushServer())) {
                String[] faqMqtt = initEntity.getPushServer().split(":");
                if (faqMqtt.length == 2) {
                    API.MQTT_FAQ_IP = faqMqtt[0];
                    API.MQTT_FAQ_PORT = faqMqtt[1];
                }
            }

            Const.TOGGLE_SHOW_TASK_ENTRANCE = initEntity.isShowMyTask();

            Const.TOGGLE_UPLOAD_VIDEO = initEntity.isOpenVideoUpload();
            Const.TOGGLE_CRM_TOKEN = initEntity.isSetCrmToken();
            Const.TOGGLE_OPEN_FAQ_NOTIFICATION = initEntity.isOpenPushServer();
            Const.TOGGLE_OPEN_UNREAD_MSG = initEntity.isUnreadMessage();
            Const.TOGGLE_UPLOAD_LOG = initEntity.isOpenUploadLogFile();
            Const.TOGGLE_LOCALIZE_VIA_INIT = initEntity.isLocalizeFAQViaInit();
            Const.TOGGLE_MQTT_TLS = initEntity.isTls();
            Const.TOGGLE_TRANSLATE_CS_MESSAGE = initEntity.isTranslates();
            Const.TOGGLE_DISTINGUISH_USER_BY_DEVICE = initEntity.isDistinguishUserByDevice();
            Const.TOGGLE_HIDE_BRAND = initEntity.isHideBrand();

            Const.FAQ_FILE = initEntity.getFaqFileName();
            Const.LIMIT_CHECKING_UNREAD = initEntity.getUnreadMessageTime();

            if (!TextUtils.isEmpty(initEntity.getCorrectLanguage())) {
                Const.CORRECT_LANGUAGE = initEntity.getCorrectLanguage();
            }

            if (initEntity.getNetworkCheckSetting() != null) {
                Const.TOGGLE_NET_CHECK = true;
                Const.NET_TRACE_ROUTE = initEntity.getNetworkCheckSetting().getTraceroute();
                Const.NET_PING = initEntity.getNetworkCheckSetting().getPing();
            }

            if (initEntity.getSdkCustomConfig() != null) {
                InitEntity.SdkCustomConfigEntity sdkCustomConfig = initEntity.getSdkCustomConfig();
                API.CONFIG_INIT_PROCESS = sdkCustomConfig.getProcess();
                API.CONFIG_INIT_TEXT = sdkCustomConfig.getText();
                API.CONFIG_INIT_BUSINESS_LOGIC = sdkCustomConfig.getGeneral();
            }

            Const.SAMPLING_RATE_VALUE = 100;

            ResourceTracker.markInitEnded(true);
            LocalizeHelper.goFetchLocalizeData(LocalizeHelper.FETCH_SOURCE_INIT);

            if (requireCallback) {
                Log.e("AIHelp", "AIHelp is now ready to help you have conversations with your users!");
            }
        }
    }

    @Override
    public void onRetryRequest() {
        requestInit(true);
    }

    @Override
    public void onRetryUpToMaxCount(int code, String errorMsg) {
        onAIHelpInitializedCallback(false, errorMsg);
    }
}
