package net.aihelp.ui.adapter;

import android.content.Context;
import android.os.Build;
import android.text.Layout;
import android.text.Spannable;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.text.style.AlignmentSpan;
import android.text.style.ForegroundColorSpan;
import android.util.TypedValue;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.LinearLayout;
import android.widget.TextView;

import net.aihelp.common.CustomConfig;
import net.aihelp.data.model.cs.storyline.BotBillEntity;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import androidx.appcompat.widget.AppCompatRadioButton;

public class BillingListAdapter extends BaseAdapter {

    private final Context mContext;
    private final boolean isEnableSend;
    private final List<BotBillEntity> mBillingList = new ArrayList<>();
    private BotBillEntity mCurrentCheckedEntity;

    public BillingListAdapter(Context context, boolean isEnableSend) {
        this.isEnableSend = isEnableSend;
        this.mContext = context;
    }

    public void update(JSONArray jsonArray) {
        try {
            for (int i = 0; i < jsonArray.length(); i++) {
                mBillingList.add(new BotBillEntity(jsonArray.optString(i)));
            }
            notifyDataSetChanged();
        } catch (Exception e) {
            // ignored
        }
    }

    private TextView getBillInfo(String jsonString, String key) {
        TextView textView = new TextView(mContext);
        textView.setTextSize(TypedValue.COMPLEX_UNIT_SP, 14);
        textView.setPadding(0, 7, 0, 7);
        try {
            JSONObject billItem = new JSONObject(jsonString);
            String value = billItem.optString(key);

            ForegroundColorSpan keySpan = new ForegroundColorSpan(Styles.getColorWithAlpha(CustomConfig.CommonSetting.textColor, 0.5));
            ForegroundColorSpan valueSpan = new ForegroundColorSpan(Styles.getColor(CustomConfig.CommonSetting.textColor));

            SpannableStringBuilder builder = new SpannableStringBuilder();


            builder.append(key).append(": ");
            builder.setSpan(keySpan, 0, key.length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);

            builder.append(value);
            builder.setSpan(valueSpan, key.length(), key.length() + 1 + ": ".length() + value.length() - 1,
                    Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);

//            builder.setSpan(new AlignmentSpan.Standard(Layout.Alignment.ALIGN_OPPOSITE), 0, builder.length(), Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                textView.setTextDirection(View.TEXT_DIRECTION_LOCALE);
            }
            textView.setText(builder);

        } catch (Exception e) {
            // ignored
        }
        return textView;
    }

    private ArrayList<String> getSortedKeys(String s) {
        ArrayList<String> keyList = new ArrayList<>();
        try {
            JSONObject billItem = new JSONObject(s);
            Iterator<String> iterator = billItem.keys();
            while (iterator.hasNext()) {
                keyList.add(iterator.next());
            }
        } catch (Exception e) {
            // ignored
        }
        return keyList;
    }

    @Override
    public int getCount() {
        return mBillingList.size();
    }

    @Override
    public BotBillEntity getItem(int position) {
        return mBillingList.size() == 0 ? null : mBillingList.get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public View getView(int position, View convertView, ViewGroup parent) {
        BillingListAdapter.ViewHolder holder;
        if (convertView == null) {
            convertView = View.inflate(mContext, ResResolver.getLayoutId("aihelp_ada_billing_list"), null);
            holder = new BillingListAdapter.ViewHolder(convertView);
            convertView.setTag(holder);
        } else {
            holder = (BillingListAdapter.ViewHolder) convertView.getTag();
        }

        BotBillEntity botBillEntity = mBillingList.get(position);

        holder.vDivider.setVisibility(position == 0 ? View.GONE : View.VISIBLE);
        holder.vDivider.setBackgroundColor(Styles.getColorWithAlpha(CustomConfig.CommonSetting.textColor, 0.1));

        holder.radioButton.setBackground(Styles.makeSelector(mContext));
        holder.radioButton.setChecked(botBillEntity.isChecked());

        holder.billContainer.removeAllViews();

        ArrayList<String> sortedKeys = getSortedKeys(botBillEntity.getOriginJson());
        for (String key : sortedKeys) {
            holder.billContainer.addView(getBillInfo(botBillEntity.getOriginJson(), key));
        }

        if (isEnableSend) {
            convertView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (mCurrentCheckedEntity != null) {
                        mCurrentCheckedEntity.setChecked(false);
                    }
                    botBillEntity.setChecked(true);
                    mCurrentCheckedEntity = botBillEntity;
                    notifyDataSetChanged();
                    if (listener != null) {
                        listener.onOrderChecked();
                    }
                }
            });
        } else {
            holder.radioButton.setVisibility(View.GONE);
        }

        return convertView;
    }

    public String getCheckedBill() {
        if (mCurrentCheckedEntity != null) {
            return mCurrentCheckedEntity.getOriginJson();
        }
        return "";
    }

    private static class ViewHolder {

        private final View vDivider;
        private final LinearLayout billContainer;
        private final AppCompatRadioButton radioButton;

        private ViewHolder(View convertView) {
            vDivider = convertView.findViewById(ResResolver.getViewId("aihelp_bill_divider"));
            billContainer = convertView.findViewById(ResResolver.getViewId("aihelp_ll_bill"));
            radioButton = convertView.findViewById(ResResolver.getViewId("aihelp_rb_bill"));
        }

    }

    public interface OnOrderCheckedListener {
        void onOrderChecked();
    }

    private OnOrderCheckedListener listener;

    public void setOnOrderCheckedListener(OnOrderCheckedListener listener) {
        this.listener = listener;
    }
}
