package net.aihelp.ui.adapter.task;

import android.annotation.SuppressLint;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ProgressBar;
import android.widget.TextView;

import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.data.event.PageHoppingEvent;
import net.aihelp.data.model.task.TaskEntity;
import net.aihelp.utils.DateFormatUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.core.content.ContextCompat;
import androidx.recyclerview.widget.RecyclerView;

public class TaskListAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder> {

    private final int VIEW_TYPE_ITEM = 0;
    private final int VIEW_TYPE_LOADING = 1;

    public List<TaskEntity> mTaskList = new ArrayList<>();

    public List<TaskEntity> getTaskList() {
        return mTaskList;
    }

    @NonNull
    @Override
    public RecyclerView.ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        if (viewType == VIEW_TYPE_ITEM) {
            View view = LayoutInflater.from(parent.getContext()).inflate(ResResolver.getLayoutId("aihelp_ada_task_list"), parent, false);
            return new TaskViewHolder(view);
        } else {
            View view = LayoutInflater.from(parent.getContext()).inflate(ResResolver.getLayoutId("aihelp_ada_loading_more"), parent, false);
            return new LoadingViewHolder(view);
        }
    }

    @Override
    public void onBindViewHolder(@NonNull RecyclerView.ViewHolder viewHolder, int position) {

        if (viewHolder instanceof TaskViewHolder) {
            populateItemRows((TaskViewHolder) viewHolder, position);
        } else if (viewHolder instanceof LoadingViewHolder) {
            showLoadingView((LoadingViewHolder) viewHolder, position);
        }

    }

    @Override
    public int getItemCount() {
        return mTaskList == null ? 0 : mTaskList.size();
    }

    @Override
    public int getItemViewType(int position) {
        return mTaskList != null && mTaskList.get(position) == null ? VIEW_TYPE_LOADING : VIEW_TYPE_ITEM;
    }

    private void showLoadingView(LoadingViewHolder viewHolder, int position) {
        //ProgressBar would be displayed
    }

    private void populateItemRows(TaskViewHolder viewHolder, int position) {
        TaskEntity taskEntity = mTaskList.get(position);
        if (taskEntity != null) {

            viewHolder.vUnreadStatus.setBackground(Styles.getDrawable(Styles.getColor(CustomConfig.CommonSetting.highlightedColor), 999));
            viewHolder.vUnreadStatus.setVisibility(taskEntity.isUnRead() ? View.VISIBLE : View.GONE);

            Styles.reRenderTextView(viewHolder.tvName, taskEntity.getTitle());

            Styles.reRenderTextView(viewHolder.tvTime, DateFormatUtil.dateFormat(taskEntity.getCreateTime(), null),
                    Styles.getColor(CustomConfig.CommonSetting.textColor), true, 13
            );

            Styles.reRenderTextView(viewHolder.tvStatus, taskEntity.getStatus());

            viewHolder.itemView.setBackground(getBackgroundDrawable());
            viewHolder.itemView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    taskEntity.setUnRead(false);
                    notifyItemChanged(position);

                    // go to content fragment
                    Bundle bundle = new Bundle();
                    bundle.putString(IntentValues.TASK_CODE, taskEntity.getTaskCode());
                    EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_TICKET_CONTENT, bundle));

//                    String localeParams = String.format("title=%s&taskstatus=%s&replycontent=%s&taskcontent=%s",
//                            ResResolver.getString("aihelp_my_task"),
//                            ResResolver.getString("aihelp_task_status"),
//                            ResResolver.getString("aihelp_reply_content"),
//                            ResResolver.getString("aihelp_task_content")
//                    );
//                    String contentUrl = String.format(
//                            "https://%s/workorderDetail/index.html?appId=%s&taskCode=%s&lan=%s&userId=%s&%s&isCustom=1",
//                            API.HOST_URL, Const.APP_ID, taskEntity.getTaskCode(), Const.CORRECT_LANGUAGE, UserProfile.USER_ID, localeParams);
//                    Bundle bundle = new Bundle();
//                    bundle.putString(IntentValues.INTENT_URL, contentUrl);
//                    EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_FORM, bundle));
                }
            });
        }
    }

    @SuppressLint("NotifyDataSetChanged")
    public void updateStatusCount(List<TaskEntity> taskList) {
        this.mTaskList.addAll(taskList);
        notifyDataSetChanged();
    }

    public void onLoadMore() {
        mTaskList.add(null);
        notifyItemInserted(mTaskList.size() - 1);
    }

    private static class TaskViewHolder extends RecyclerView.ViewHolder {

        View itemView;
        View vUnreadStatus;
        TextView tvName;
        TextView tvTime;
        TextView tvStatus;

        public TaskViewHolder(@NonNull View itemView) {
            super(itemView);
            this.itemView = itemView;
            vUnreadStatus = itemView.findViewById(ResResolver.getViewId("aihelp_v_unread_status"));
            tvName = itemView.findViewById(ResResolver.getViewId("aihelp_tv_name"));
            tvTime = itemView.findViewById(ResResolver.getViewId("aihelp_tv_time"));
            tvStatus = itemView.findViewById(ResResolver.getViewId("aihelp_tv_status"));
        }
    }

    private static class LoadingViewHolder extends RecyclerView.ViewHolder {

        ProgressBar progressBar;

        public LoadingViewHolder(@NonNull View itemView) {
            super(itemView);
            itemView.setBackground(getBackgroundDrawable());
            progressBar = itemView.findViewById(ResResolver.getViewId("aihelp_progress_bar"));
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
                Drawable drawable = progressBar.getIndeterminateDrawable().mutate();
                drawable.setColorFilter(ContextCompat.getColor(itemView.getContext(), ResResolver.getColorId("aihelp_color_main")), PorterDuff.Mode.SRC_IN);
                progressBar.setIndeterminateDrawable(drawable);
            }
        }
    }

    private static Drawable getBackgroundDrawable() {
        int colorFromHex = Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha);
        return Styles.getDrawable(colorFromHex, 0);
    }
}
