package net.aihelp.ui.cs.middle.intent;

import android.content.Context;
import android.graphics.Color;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import net.aihelp.common.CustomConfig;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.bus.Subscribe;
import net.aihelp.core.util.bus.ThreadMode;
import net.aihelp.core.util.bus.event.EventCenter;
import net.aihelp.data.event.OrientationChangeEvent;
import net.aihelp.data.localize.config.ProcessEntranceHelper;
import net.aihelp.data.model.config.IntentEntity;
import net.aihelp.data.model.config.ProcessEntity;
import net.aihelp.data.track.statistic.StatisticTracker;
import net.aihelp.ui.adapter.SmartIntentAdapter;
import net.aihelp.ui.cs.util.TicketStatusTracker;
import net.aihelp.ui.widget.AIHelpButton;
import net.aihelp.utils.ListUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.SoftInputUtil;
import net.aihelp.utils.Styles;

import java.util.List;

import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class SmartIntentView extends LinearLayout implements View.OnClickListener,
        SmartIntentAdapter.OnIntentSelectedListener {

    private ProcessEntity currentProcess;
    private final SmartIntentScrollView svIntent;
    private final AIHelpButton tvIntent;

    private final SmartIntentAdapter subIntentAdapter;
    private IOnIntentViewEventListener intentViewEventListener;

    public void setOnIntentViewEventListener(IOnIntentViewEventListener intentViewEventListener) {
        this.intentViewEventListener = intentViewEventListener;
    }

    public SmartIntentView(Context context) {
        this(context, null);
    }

    public SmartIntentView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public SmartIntentView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        View.inflate(context, ResResolver.getLayoutId("aihelp_smart_intent_view"), this);

        svIntent = findViewById(ResResolver.getViewId("aihelp_intent_scroll"));
        int backgroundColor = Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                Math.min(CustomConfig.CommonSetting.upperBackgroundAlpha + 0.6, 0.8));
        svIntent.setBackgroundColor(backgroundColor);

        tvIntent = findViewById(ResResolver.getViewId("aihelp_tv_intent"));
        tvIntent.setMaxWidth((int) (Styles.getScreenWidth(getContext()) * 0.6f));

        tvIntent.setOnClickListener(this);

        View rootDivider = findViewById(ResResolver.getViewId("aihelp_v_divider_root"));
        View subDivider = findViewById(ResResolver.getViewId("aihelp_v_divider_sub"));
        rootDivider.setBackgroundColor(Styles.getColorWithAlpha(CustomConfig.CommonSetting.textColor, 0.1f));
        subDivider.setBackgroundColor(Styles.getColorWithAlpha(CustomConfig.CommonSetting.textColor, 0.1f));

        subIntentAdapter = new SmartIntentAdapter(getContext(), this);

        setVisibility(GONE);
    }

    public void updateViewVisibility() {
        currentProcess = ProcessEntranceHelper.INSTANCE.getCurrentProcess();
        if (tvIntent == null || svIntent == null
                || TextUtils.isEmpty(currentProcess.getSmartIntentName())
                || ListUtil.isListEmpty(currentProcess.getSmartIntentList())) {
            setVisibility(GONE);
            return;
        }

        tvIntent.setText(currentProcess.getSmartIntentName());
        tvIntent.setMaxWidth((int) (Styles.getScreenWidth(getContext()) * 0.6f));

        if (TicketStatusTracker.isTicketServingByAgent() || TicketStatusTracker.isTicketServingByRPA()) {
            setVisibility(GONE);
            tvIntent.setVisibility(GONE);
            svIntent.setVisibility(GONE);
        } else {
            if (TicketStatusTracker.isTicketActive && TicketStatusTracker.isTicketServingByAnswerBot()) {
                if (currentProcess != null && currentProcess.isEnableInteraction() &&
                        !ListUtil.isListEmpty(currentProcess.getSmartIntentList())) {
                    setVisibility(VISIBLE);
                    tvIntent.setVisibility(VISIBLE);
                    svIntent.setVisibility(GONE);
                } else {
                    setVisibility(GONE);
                    tvIntent.setVisibility(GONE);
                    svIntent.setVisibility(GONE);
                }
            } else {
                // Fold the intent menu if it's landscape view, show the menu list when it's first displayed otherwise
                if (Styles.isLandscape()) {
                    setVisibility(VISIBLE);
                    tvIntent.setVisibility(VISIBLE);
                    svIntent.setVisibility(GONE);
                } else {
                    showRootIntents();
                }
            }
        }
    }

    public void showRootIntents() {
        if (currentProcess != null) {
            List<IntentEntity> smartIntentList = currentProcess.getSmartIntentList();
            if (!ListUtil.isListEmpty(smartIntentList)) {
                setVisibility(VISIBLE);

                subIntentAdapter.clear();
                tvIntent.setVisibility(View.GONE);
                svIntent.setVisibility(View.VISIBLE);
                svIntent.scrollIntentTo(0);

                TextView tvTitle = findViewById(ResResolver.getViewId(("aihelp_tv_root_title")));
                tvTitle.setTextColor(Styles.getColor(CustomConfig.CommonSetting.textColor));

                RecyclerView rvIntent = findViewById(ResResolver.getViewId("aihelp_rv_root_intent"));
                rvIntent.setLayoutManager(new LinearLayoutManager(getContext()));
                SmartIntentAdapter actionAdapter = new SmartIntentAdapter(getContext(), this);
                rvIntent.setAdapter(actionAdapter);
                actionAdapter.update(smartIntentList);

                ImageView ivClose = findViewById(ResResolver.getViewId("aihelp_iv_root_close"));
                Styles.reRenderImageView(ivClose, "aihelp_svg_ic_close_dialog");
                ivClose.setOnClickListener(this);
            } else {
                setVisibility(GONE);
            }
        }
    }

    public void showSubIntents(IntentEntity entity) {
        ImageView ivBack = findViewById(ResResolver.getViewId("aihelp_iv_sub_back"));
        Styles.reRenderImageView(ivBack, "aihelp_svg_ic_back");
        ivBack.setOnClickListener(this);

        TextView tvTitle = findViewById(ResResolver.getViewId(("aihelp_tv_sub_title")));
        Styles.reRenderTextView(tvTitle, entity.getIntentName(), Color.parseColor(CustomConfig.CommonSetting.interactElementTextColor));
        tvTitle.setOnClickListener(this);

        RecyclerView rvSubIntent = findViewById(ResResolver.getViewId("aihelp_rv_sub_intent"));
        rvSubIntent.setLayoutManager(new LinearLayoutManager(getContext()));
        rvSubIntent.setAdapter(subIntentAdapter);
        subIntentAdapter.update(entity.getIntentList());

        ImageView ivClose = findViewById(ResResolver.getViewId("aihelp_iv_sub_close"));
        Styles.reRenderImageView(ivClose, "aihelp_svg_ic_close_dialog");
        ivClose.setOnClickListener(this);
    }

    @Override
    public void onClick(View v) {
        if (v.getId() == ResResolver.getViewId("aihelp_tv_intent")) {
            showRootIntents();
        }

        if (v.getId() == ResResolver.getViewId("aihelp_iv_root_close") ||
                v.getId() == ResResolver.getViewId("aihelp_iv_sub_close")) {
            if (currentProcess != null) {
                tvIntent.setVisibility(currentProcess.isEnableInteraction() ? View.VISIBLE : View.GONE);
                svIntent.setVisibility(View.GONE);
                subIntentAdapter.clear();
                svIntent.scrollIntentTo(0);
            }
        }

        if (v.getId() == ResResolver.getViewId("aihelp_iv_sub_back") || v.getId() == ResResolver.getViewId("aihelp_tv_sub_title")) {
            subIntentAdapter.clear();
            svIntent.scrollIntentTo(0);
        }
    }

    @Override
    public void onIntentSelected(IntentEntity intent, boolean hasNext) {
        if (hasNext) {
            showSubIntents(intent);
            svIntent.scrollIntentTo(Integer.MAX_VALUE);
            return;
        }
        if (intentViewEventListener != null && intent != null) {
            SoftInputUtil.hideSoftInput(getContext(), svIntent);
            intentViewEventListener.onIntentSelected(intent);
            StatisticTracker.getInstance().selectSmartIntent(String.valueOf(intent.getIntentId()));
            svIntent.setVisibility(View.GONE);
        }
    }

    @Override
    public void setVisibility(int visibility) {
        super.setVisibility(visibility);
        if (intentViewEventListener != null) {
            intentViewEventListener.onIntentViewVisibilityChanged(visibility);
        }
    }

    // region - 处理屏幕旋转相关业务
    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        EventBus.getDefault().register(this);
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        EventBus.getDefault().unregister(this);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEventComing(EventCenter event) {
        if (event instanceof OrientationChangeEvent) {
            post(new Runnable() {
                @Override
                public void run() {
                    updateViewVisibility();
                }
            });
        }
    }
    // endregion

}
