package net.aihelp.ui.cs.util.rpa.factory;

import android.text.TextUtils;

import net.aihelp.data.model.rpa.msg.base.FileMessage;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.model.rpa.msg.UserMessage;
import net.aihelp.utils.RegexUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class UserMessageFactory extends MessageFactory {

    public static List<Message> getUserMessage(String msg, long timestamp) {
        switch (getUserMessageType(msg)) {
            case Message.TYPE_USER_IMAGE:
                return getUserImageMessage(msg, timestamp);
            case Message.TYPE_USER_VIDEO:
                return getUserVideoMessage(msg, timestamp);
            case Message.TYPE_USER_FILE:
                return getUserFileMessage(msg, timestamp);
            default:
                return getUserTextMessage(msg, timestamp);
        }
    }

    private static int getUserMessageType(String message) {
        if (isUserImage(message)) {
            return Message.TYPE_USER_IMAGE;
        }
        if (isUserVideo(message)) {
            return Message.TYPE_USER_VIDEO;
        }
        if (isUserAttachment(message)) {
            return Message.TYPE_USER_FILE;
        }
        return Message.TYPE_USER_TEXT;
    }

    private static List<Message> getUserTextMessage(String msg, long timestamp) {
        ArrayList<Message> msgList = new ArrayList<>();
        UserMessage userMessage = new UserMessage();
        String msgContent = getPeeledContent(msg);
        userMessage.setTimestamp(timestamp);
        userMessage.setContent(msgContent);
        msgList.add(userMessage);
        replaceMediaUrls(msgContent, msgList, userMessage);
        return msgList;
    }

    private static List<Message> getUserImageMessage(String msg, long timestamp) {
        ArrayList<Message> msgList = new ArrayList<>();
        FileMessage imageMessage = new FileMessage(Message.TYPE_USER_IMAGE, msg);
        if (isUserAttachment(msg)) {
            imageMessage = getAttachmentMessage(msg, Message.TYPE_USER_IMAGE);
        }
        imageMessage.setTimestamp(timestamp);
        msgList.add(imageMessage);
        return msgList;
    }

    private static List<Message> getUserVideoMessage(String msg, long timestamp) {
        ArrayList<Message> msgList = new ArrayList<>();
        FileMessage videoMessage = new FileMessage(Message.TYPE_USER_VIDEO, msg);
        if (isUserAttachment(msg)) {
            videoMessage = getAttachmentMessage(msg, Message.TYPE_USER_VIDEO);
        }
        videoMessage.setTimestamp(timestamp);
        msgList.add(videoMessage);
        return msgList;
    }

    private static List<Message> getUserFileMessage(String msg, long timestamp) {
        ArrayList<Message> msgList = new ArrayList<>();
        FileMessage fileMessage = getAttachmentMessage(msg, Message.TYPE_USER_FILE);
        fileMessage.setTimestamp(timestamp);
        fileMessage.setNormalMessage(true);
        msgList.add(fileMessage);
        return msgList;
    }

    private static boolean isUserText(String message) {
        return !TextUtils.isEmpty(getPeeledContent(message));
    }

    private static boolean isUserImage(String message) {
        return Pattern.compile(RegexUtil.REGEX_IMAGE).matcher(message).matches() || isAttachmentImage(message);
    }

    private static boolean isUserVideo(String message) {
        return Pattern.compile(RegexUtil.REGEX_VIDEO).matcher(message).matches() || isAttachmentVideo(message);
    }

    private static boolean isUserAttachment(String message) {
        return hasOnlyOneAttachment(message);
    }

    public static List<Message> getUserFormMsgList(String msg) {
        ArrayList<Message> formMsgList = new ArrayList<>();
        UserMessage userMessage = new UserMessage();
        userMessage.setContent(msg);
        formMsgList.add(userMessage);
        // only if text type
        replaceMediaUrls(msg, formMsgList, userMessage);
        return formMsgList;
    }

    private static void replaceMediaUrls(String msg, ArrayList<Message> formMsgList, Message message) {
        int fakeTimeStamp = 10; // 为匹配中的图片或视频消息增加几毫秒，保证消息顺序（其实没用，因为 List 本身就有序）
        String[] messageSplits = msg.split(";");
        for (String item : messageSplits) {
            Matcher replaceImageMatcher = Pattern.compile(RegexUtil.REGEX_IMAGE).matcher(item.trim());

            // 匹配图片并生成图片消息
            while (replaceImageMatcher.find()) {
                String imageUrl = replaceImageMatcher.group();
                FileMessage formImageMsg = new FileMessage(Message.TYPE_USER_IMAGE, imageUrl);
                formImageMsg.setTimestamp(message.getTimestamp() + fakeTimeStamp++);
                formMsgList.add(formImageMsg);
                // 添加消息成功后就移除对应的链接
                msg = msg.replace(imageUrl, "");
            }

            // 使用移除 image 后的字符串重新尝试匹配 videoUrl
            Matcher replaceVideoMatcher = Pattern.compile(RegexUtil.REGEX_VIDEO).matcher(msg);
            while (replaceVideoMatcher.find()) {
                String videoUrl = replaceVideoMatcher.group();
                FileMessage formVideoMsg = new FileMessage(Message.TYPE_USER_VIDEO, videoUrl);
                formVideoMsg.setTimestamp(message.getTimestamp() + fakeTimeStamp++);
                formMsgList.add(formVideoMsg);
                msg = msg.replace(videoUrl, "");
            }
        }
    }

}
