package net.aihelp.ui.cs.util.rpa.helper;

import android.text.TextUtils;

import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.data.model.cs.ChatInfoEntity;
import net.aihelp.data.model.rpa.msg.BotMessage;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.ui.cs.util.rpa.factory.AgentMessageFactory;
import net.aihelp.ui.cs.util.rpa.factory.BotMessageFactory;
import net.aihelp.ui.cs.util.rpa.factory.TimeMessageFactory;
import net.aihelp.ui.cs.util.rpa.factory.UserMessageFactory;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class HistoryHelper {

    public static final int HISTORY_TYPE_LOGIN = 1;
    public static final int HISTORY_TYPE_REFRESH = 2;
    public static final int HISTORY_TYPE_POLL = 3;

    /**
     * 返回一条消息加工成 model 后的数据集合
     * @param chatInfoEntity
     * @param historyType
     * @return 由于图片和视频可能混在 msg 中，需要将其拆出来单独展示成一条消息，所以返回的可能是个列表
     */
    private static List<Message> getListFromMsgString(ChatInfoEntity chatInfoEntity, int historyType) {
        int msgType = chatInfoEntity.getMsgType();
        long timeStamp = chatInfoEntity.getTimeStamp();
        String msg = chatInfoEntity.getMessage();

        ArrayList<Message> messageList = new ArrayList<>();
        switch (msgType) {
            case ChatInfoEntity.TYPE_TIMESTAMP:
                if (historyType == HISTORY_TYPE_POLL) break; // ignore the timestamp returned by poll api
                messageList.add(TimeMessageFactory.getTimeMessage(timeStamp));
                break;
            case ChatInfoEntity.TYPE_AGENT:
                ChatInfoHelper.prepareNickname(chatInfoEntity);
                messageList.add(AgentMessageFactory.getAgentMessage(msg, timeStamp, chatInfoEntity.getAgentNickname()));
                break;
            case ChatInfoEntity.TYPE_BOT:
                JSONObject msgJsonObject = JsonHelper.getJsonObject(msg);
                chatInfoEntity.setUserId(msgJsonObject.optInt("customerId"));
                ChatInfoHelper.prepareNickname(chatInfoEntity);
                BotMessage botMessage = BotMessageFactory.getHistoricBotMessage(msg, timeStamp,
                        chatInfoEntity.getAgentNickname(), historyType);
                if (botMessage.hasSelfService() && botMessage.getSelfService().isEnableSend()) {
                    // 允许发送的自助服务属于 RPA 流程，会显示在 BottomView 的位置，不需要加入列表，直接把列表返回
                    return messageList;
                }
                messageList.add(botMessage);
                break;
            default:
                messageList.addAll(UserMessageFactory.getUserMessage(msg, timeStamp));
                break;
        }
        return messageList;
    }

    public static List<Message> getHistoryList(String msgJson, int historyType) {
        ArrayList<Message> list = new ArrayList<>();
        if (!TextUtils.isEmpty(msgJson)) {
            for (ChatInfoEntity chatInfoEntity : ChatInfoHelper.getChatInfoList(msgJson, historyType)) {
                list.addAll(getListFromMsgString(chatInfoEntity, historyType));
            }
        }
        return list;
    }

}
