package net.aihelp.ui.cs.util.rpa.helper;

import android.text.TextUtils;

import net.aihelp.BuildConfig;
import net.aihelp.common.API;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.UserProfile;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.common.Const;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.data.localize.config.FeedbackOpinionHelper;
import net.aihelp.data.model.rpa.step.RPAStep;
import net.aihelp.data.model.rpa.msg.base.Message;
import net.aihelp.data.track.statistic.StatisticTracker;
import net.aihelp.ui.cs.util.TicketStatusTracker;
import net.aihelp.ui.cs.util.rpa.factory.AgentMessageFactory;
import net.aihelp.ui.cs.util.rpa.factory.BotMessageFactory;
import net.aihelp.ui.cs.util.rpa.factory.MessageFactory;
import net.aihelp.utils.DeviceInfoUtil;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class ResponseHelper {

    public static final int RPA_PROCEDURE_TICKET_CREATED = 1;
    public static final int RPA_PROCEDURE_TICKET_RESET = 2;

    public static final int RPA_PROCEDURE_BOT_NORMAL = 1001;
    public static final int RPA_PROCEDURE_BOT_STOP_AND_WAIT = 1002;

    public static final int RPA_PROCEDURE_AGENT_ALREADY_ASSIGNED = 2001;

    //region - 获取 RPA Message
    public static Message getRPAMessage(String response) {
        if (TextUtils.isEmpty(response)) {
            Message message = new Message(Message.TYPE_AGENT_TEXT);
            message.setNormalMessage(false);
            return message;
        }

        try {
            JSONObject jsonObject = new JSONObject(response);

            boolean isAgent = jsonObject.has("usertype") && jsonObject.optInt("usertype") == 0;
            String nickname = JsonHelper.optString(jsonObject, "nickname");
            long timestamp = jsonObject.optLong("time");

            if (isAgent) {
                String msg = JsonHelper.optString(jsonObject, "msg");
                return AgentMessageFactory.getAgentMessage(msg, timestamp, nickname);
            } else {
                JSONObject bot = JsonHelper.getJsonObject(jsonObject, "bot");
                return BotMessageFactory.getResponseBotMessage(bot, timestamp, nickname);
            }
        } catch (Exception e) {
            e.printStackTrace();
            Message message = new Message(Message.TYPE_AGENT_TEXT);
            message.setNormalMessage(false);
            return message;
        }
    }
    //endregion

    //region - 获取 RPA Step 对象
    public static RPAStep getRPAStep(String response) {
        RPAStep rpaStep = new RPAStep();

        // 人工客服消息或者收到新消息的时候客诉已经是待评价状态，那就不再操作底部输入框
        boolean isTicketAlreadyFinished = TicketStatusTracker.isTicketFinished &&
                (TicketStatusTracker.isTicketWaitForAskingResolveStatus || TicketStatusTracker.isTicketWaitForRating);
        if (MessageFactory.isAgentMsg(response) || isTicketAlreadyFinished) {
            rpaStep.setNextStep(RPAStep.STEP_IGNORE_THIS);
            return rpaStep;
        }

        try {
            JSONObject jsonObject = new JSONObject(response);
            JSONObject bot = JsonHelper.getJsonObject(jsonObject, "welcomeBot");
            if (bot.length() == 0) {
                bot = JsonHelper.getJsonObject(jsonObject, "bot");
            }
            int status = bot.optInt("status", -1);
            switch (status) {
                case RPA_PROCEDURE_BOT_STOP_AND_WAIT:
                    rpaStep.setNextStep(RPAStep.STEP_STOP_AND_WAIT);
                    break;
                case RPA_PROCEDURE_TICKET_RESET:
                    rpaStep.setNextStep(RPAStep.STEP_NEW_CONVERSATION);
                    break;
                case RPA_PROCEDURE_TICKET_CREATED: // 客诉生成时忽略底部区域的操作
                    Const.TOGGLE_FETCH_MESSAGE = true;
                    TicketStatusTracker.isTicketActive = true;
                    TicketStatusTracker.currentTicketId = JsonHelper.optString(jsonObject, "ticketId");
                case RPA_PROCEDURE_AGENT_ALREADY_ASSIGNED:
                    rpaStep.setNextStep(RPAStep.STEP_IGNORE_THIS);
                    break;
                default:
                    JSONObject botStep = JsonHelper.getJsonObject(bot, "step");
                    if (botStep.keys().hasNext()) {
                        // 单独封装控制底部布局相关的数据
                        // 当前步骤 id、是否允许跳过、是否允许上传日志、下一步的用户行为、跳过文案、是否允许在 action 阶段手动输入
                        rpaStep.setStepId(JsonHelper.optString(botStep, "stepId"));
                        rpaStep.setEnablePrevStep(botStep.optInt("allowBack") == 1);
                        rpaStep.setEnableSkip(botStep.optInt("allowSkip") == 1);
                        rpaStep.setEnableUpload(botStep.optInt("autoUpload") == 1);
                        rpaStep.setNextStep(botStep.optInt("nextAction"));
                        rpaStep.setSkipHint(JsonHelper.optString(botStep, "skipLabel"));
                        rpaStep.setPrevStepHint(JsonHelper.optString(botStep, "backLabel"));
                        rpaStep.setEnableActionInput(botStep.optInt("allowInputBox") == 1);
                        rpaStep.setAttachmentTypes(JsonHelper.optString(botStep, "attachmentsTypes"));

                        // 获取 Action 列表相关的数据
                        JSONArray actionArray = JsonHelper.getJsonArray(botStep, "buttons");
                        List<RPAStep.Action> actionList = new ArrayList<>();
                        for (int i = 0; i < actionArray.length(); i++) {
                            JSONObject action = JsonHelper.getJsonObject(actionArray, i);
                            int order = action.optInt("order");
                            String actionId = JsonHelper.optString(action, "id");
                            String actionContent = JsonHelper.optString(action, "value");
                            actionList.add(RPAStep.Action.getInstance(order, actionId, actionContent));
                        }
                        rpaStep.setActionList(actionList);
                    }
                    break;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return rpaStep;
    }
    //endregion

    //region - 发送 ack，通知后端推送 mqtt
    public static void notifyMqttPush(String response) {
        try {
            JSONObject mqttNotifyParams = new JSONObject();
            mqttNotifyParams.put("timestamp", System.currentTimeMillis());
            if (!TextUtils.isEmpty(response)) {
                mqttNotifyParams.put("responseId", JsonHelper.optString(new JSONObject(response), "responseId"));
            }
            AIHelpRequest.getInstance().requestGetByAsync(API.NOTIFY_MQTT_PUSH, mqttNotifyParams, null);
        } catch (Exception e) {
            // ignored
        }
    }
    // endregion

    // region - 解析 login & polling 数据
    public static List<Message> getLoginResponse(String response) {
        final ArrayList<Message> list = new ArrayList<>();
        JSONObject loginObj = JsonHelper.getJsonObject(response);

        // 添加欢迎语
        list.add(Message.getDefaultMessage());

        // 默认认为没有历史记录，初始化首条消息为当前时间
        TicketStatusTracker.firstMessageTimeStampInList = System.currentTimeMillis();

        // 客诉历史记录
        List<Message> historyList = getHistoryMessageList(response, HistoryHelper.HISTORY_TYPE_LOGIN);
        TicketStatusTracker.setFirstMessageTimeStampInList(historyList);
        list.addAll(historyList);

        // 开关类配置
        updateTicketStatusTracker(response);

        // 玩家评价数据
        if (JsonHelper.hasKey(loginObj, "evaluation")) {
            String evaluation = JsonHelper.optString(loginObj, "evaluation");
            if (!TextUtils.isEmpty(evaluation)) {
                JSONArray jsonArray = JsonHelper.getJsonArray(evaluation);
                FeedbackOpinionHelper.INSTANCE.prepareDataSource(jsonArray);
            }
        }

        // 在 MQTT 断开后轮询调用接口，获取断开期间的客服消息
        if (JsonHelper.hasKey(loginObj, "expandField")) {
            JSONObject expandField = JsonHelper.getJsonObject(loginObj, "expandField");
            TicketStatusTracker.ticketPollInterval = expandField.optInt("ticketPollInterval", 60);
        }
        return list;
    }

    public static List<Message> getPollResponse(String response) {
        updateTicketStatusTracker(response);
        return getHistoryMessageList(response, HistoryHelper.HISTORY_TYPE_POLL);
    }
    // endregion

    // region - 其它处理逻辑，工具类方法

    /**
     * 客诉历史记录
     * 1、根据是否有历史记录来决定当前客诉是否处于进行中状态
     * 2、插入当前客诉的历史消息
     */
    private static List<Message> getHistoryMessageList(String response, int historyType) {
        JSONObject jsonObject = JsonHelper.getJsonObject(response);
        if (JsonHelper.hasKey(jsonObject, "chatHistoryContent")) {
            String chatHistoryContent = JsonHelper.optString(jsonObject, "chatHistoryContent");
            return HistoryHelper.getHistoryList(chatHistoryContent, historyType);
        }
        return new ArrayList<>();
    }

    /**
     * 开关类配置
     * hasTicket - 当前是否有正在进行中的客诉
     * assignType - 0 未分配，1 人工，2 RPA，3 AnswerBot
     * cisEvaluate - 当前客诉是否已结束
     * isShowResolve - 是否展示解决 & 未解决弹窗
     * isStoreReview - 是否展示 「去评分」 按钮
     * isTicketRatingEnable - 是否展示给客服评分弹窗
     * status == 9 - 客诉已被回绝
     */
    private static void updateTicketStatusTracker(String response) {
        JSONObject jsonObject = JsonHelper.getJsonObject(response);
        if (JsonHelper.hasKey(jsonObject, "chatExtraInfo")) {
            JSONObject chatExtraInfo = JsonHelper.getJsonObject(jsonObject.optString("chatExtraInfo"));
            boolean hasTicket = chatExtraInfo.optBoolean("hasTicket");
            int assignType = chatExtraInfo.optInt("assignType");
            TicketStatusTracker.updateAssignTypeWithActiveStatus(hasTicket, assignType);
            TicketStatusTracker.isTicketFinished = chatExtraInfo.optBoolean("isEvaluate");
            TicketStatusTracker.isTicketWaitForAskingResolveStatus = chatExtraInfo.optBoolean("isShowResolve");
            TicketStatusTracker.isAppRatable = chatExtraInfo.optBoolean("isStoreReview");
            TicketStatusTracker.isTicketWaitForRating = CustomConfig.InitSetting.isTicketRatingEnable;
            TicketStatusTracker.isTicketRejected = chatExtraInfo.optInt("status") == 9;
            TicketStatusTracker.currentTicketId = JsonHelper.optString(chatExtraInfo, "ticketId");
        }
    }

    public static JSONObject getLoginParams() {
        JSONObject obj = new JSONObject();
        try {
            obj.put("appId", Const.APP_ID);
            obj.put("userId", UserProfile.USER_ID);
            obj.put("userName", UserProfile.USER_NAME);
            obj.put("type", 1);
            obj.put("lan", Const.CORRECT_LANGUAGE);
            obj.put("sdkVersion", BuildConfig.SDK_VERSION);
            obj.put("source", "android");
            obj.put("gameInfo", DeviceInfoUtil.getInstance().getGameInfo().toString());
            obj.put("extendData", StatisticTracker.getInstance().getExtendDataForLogin());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return obj;
    }
    // endregion

}
