package net.aihelp.ui.faq;

import android.os.Bundle;
import android.view.View;
import android.widget.RelativeLayout;

import net.aihelp.common.ConfigValues;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.data.localize.config.ProcessEntranceHelper;
import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.data.logic.FaqPresenter;
import net.aihelp.data.model.config.ProcessEntity;
import net.aihelp.data.model.faq.FaqListEntity;
import net.aihelp.data.track.statistic.StatisticTracker;
import net.aihelp.ui.adapter.faq.FaqCardLayoutAdapter;
import net.aihelp.ui.wrapper.FaqSelectedListenerWrapper;
import net.aihelp.utils.ListUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import java.util.Iterator;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;


public class FaqListFragment extends BaseFaqFragment<FaqPresenter> {

    private String currentSectionId;
    private FaqCardLayoutAdapter mAdapter;

    public static FaqListFragment newInstance(Bundle bundle) {
        FaqListFragment faqFragment = new FaqListFragment();
        faqFragment.setArguments(bundle);
        return faqFragment;
    }

    @Override
    protected void getBundleBeforeDataPrepared(Bundle extras) {
        super.getBundleBeforeDataPrepared(extras);
        currentSectionId = extras.getString(IntentValues.SECTION_ID);
        EventTracker.INSTANCE.log(EventType.HC_PAGE_LIST_OPENED, "sectionId", currentSectionId);
    }

    @Override
    protected void initEventAndData(View mView) {

        RelativeLayout rlSearch = get("aihelp_rl_search");
        int colorFromHex = Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha);
        rlSearch.setBackgroundColor(colorFromHex);

        RecyclerView rvSubSection = get("aihelp_rv_faq_search");
        rvSubSection.setLayoutManager(new LinearLayoutManager(getContext()));
        mAdapter = new FaqCardLayoutAdapter(getContext(), true);
        mAdapter.setup(getMergedBundle(), getFaqFlowListener(), this);
        mAdapter.setOnFaqSelectedListener(new FaqSelectedListenerWrapper() {

            final Bundle bundle = getMergedBundle();

            @Override
            public void onIntentToSubSectionOrQuestionList(FaqListEntity faq) {
                bundle.putString(IntentValues.SECTION_ID, faq.getId());
                bundle.putString(IntentValues.SECTION_NAME, faq.getTitle());
                bundle.putString(IntentValues.SECTION_ICON, faq.getIconUrl());
                getFaqFlowListener().onIntentToQuestionList(bundle, true);
                handleTrackLogic(faq);
            }

            @Override
            public void onIntentToQuestionContent(FaqListEntity faq) {
                bundle.putString(IntentValues.SECTION_ID, currentSectionId);
                bundle.putString(IntentValues.SECTION_NAME, titleText);
                bundle.putString(IntentValues.SECTION_ICON, titleIcon);
                bundle.putString(IntentValues.FAQ_MAIN_ID, faq.getId());
                getFaqFlowListener().onIntentToQuestionContent(bundle, true);
            }

        });
        rvSubSection.setAdapter(mAdapter);

        // showFAQSection 的场景下准备 mqtt 和未读消息数据
        if (intentMode == IntentValues.MODE_SHOW_SECTION) {
            mPresenter.prepareFAQNotification();
        }
    }

    @Override
    protected void getBundleAfterDataPrepared(@NonNull Bundle extras) {
        String sectionId = extras.getString(IntentValues.SECTION_ID);
        mPresenter.goFetchFAQDataSource(sectionId);
    }

    @Override
    protected int getLayout() {
        return ResResolver.getLayoutId("aihelp_fra_search_faq");
    }

    @Override
    protected int getLoadingTargetViewId() {
        return ResResolver.getViewId("aihelp_rv_faq_search");
    }

    @Override
    public void refreshList(List<FaqListEntity> displayFaqList) {
        // 只有在从首页进入的时候才移除隐藏项，正常 showFAQSection 时不可以隐藏
        if (intentMode == -1 && !ListUtil.isListEmpty(displayFaqList)) {
            Iterator<FaqListEntity> iterator = displayFaqList.iterator();
            while (iterator.hasNext()) {
                FaqListEntity faqListEntity = iterator.next();
                if (faqListEntity.isHidden()) {
                    iterator.remove();
                }
            }
        }

        // 在此处进行判空处理，防止移除隐藏项后集合数据变为 0
        if (!ListUtil.isListEmpty(displayFaqList)) {
            FaqListEntity entity = displayFaqList.get(0);
            if (entity != null) {
                // titleIcon = entity.getIconUrl();
                titleText = entity.getSectionName();
            }
            mAdapter.update(displayFaqList);
        } else {
            showEmpty();
        }
    }

    private void handleTrackLogic(FaqListEntity faq) {
        ProcessEntity currentProcess = ProcessEntranceHelper.INSTANCE.getCurrentProcess();
        if (currentProcess != null && currentProcess.getIntent() == ConfigValues.ENTRANCE_FAQ_SECTION
                && currentProcess.getSectionId().equals(currentSectionId)) {
            StatisticTracker.getInstance().clickSection(faq.getId(), faq.getTitle());
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        EventTracker.INSTANCE.log(EventType.HC_PAGE_LIST_CLOSED);
    }
}
