package net.aihelp.ui.faq;

import android.os.Bundle;
import android.text.TextUtils;
import android.view.View;
import android.widget.RelativeLayout;

import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.data.track.event.EventTracker;
import net.aihelp.data.track.event.utils.EventType;
import net.aihelp.data.logic.FaqPresenter;
import net.aihelp.data.model.faq.FaqListEntity;
import net.aihelp.data.track.statistic.StatisticTracker;
import net.aihelp.ui.adapter.faq.FaqCardLayoutAdapter;
import net.aihelp.ui.wrapper.FaqSelectedListenerWrapper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.SoftInputUtil;
import net.aihelp.utils.Styles;

import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;


public class FaqSearchFragment extends BaseFaqFragment<FaqPresenter> {

    private Timer debounceTimer;
    private RecyclerView rvSearchList;
    private FaqCardLayoutAdapter mSearchAdapter;

    public static FaqSearchFragment newInstance(Bundle bundle) {
        FaqSearchFragment faqFragment = new FaqSearchFragment();
        faqFragment.setArguments(bundle);
        return faqFragment;
    }

    @Override
    protected void getBundleBeforeDataPrepared(Bundle extras) {
        super.getBundleBeforeDataPrepared(extras);
        EventTracker.INSTANCE.log(EventType.HC_PAGE_SEARCH_OPENED);
    }

    @Override
    protected void initEventAndData(View mView) {
        prepareRecyclerView();
    }

    private void prepareRecyclerView() {

        RelativeLayout rlSearch = get("aihelp_rl_search");
        int colorFromHex = Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha);
        rlSearch.setBackgroundColor(colorFromHex);

        rvSearchList = get("aihelp_rv_faq_search");
        rvSearchList.setLayoutManager(new LinearLayoutManager(getContext()));
        mSearchAdapter = new FaqCardLayoutAdapter(getContext(), false);
        mSearchAdapter.setOnFaqSelectedListener(new FaqSelectedListenerWrapper() {
            @Override
            public void onIntentToQuestionContent(FaqListEntity faq) {
                SoftInputUtil.hideSoftInput(getContext(), rvSearchList);
                Bundle bundle = getArguments() != null ? getArguments() : new Bundle();
                bundle.putString(IntentValues.FAQ_MAIN_ID, faq.getId());
                bundle.putString(IntentValues.SEARCH_MATCH, faq.getQuery());
                getFaqFlowListener().onIntentToQuestionContent(bundle, true);
            }
        });
        rvSearchList.setAdapter(mSearchAdapter);
    }

    public void onQuery(String query) {
        mPresenter.goQueryFAQList(query);
        handleTrackLogicWithDebounce(query);
    }

    private void handleTrackLogicWithDebounce(String query) {
        if (TextUtils.isEmpty(query)) return;
        if (debounceTimer != null) {
            debounceTimer.cancel();
        }
        debounceTimer = new Timer();
        debounceTimer.schedule(new TimerTask() {
            @Override
            public void run() {
                StatisticTracker.getInstance().trackSearchQuery(query);
                EventTracker.INSTANCE.log(EventType.HC_PAGE_SEARCH_QUERY, "query", query);
            }
        }, 5000);
    }

    @Override
    protected int getLayout() {
        return ResResolver.getLayoutId("aihelp_fra_search_faq");
    }

    @Override
    public void refreshList(List<FaqListEntity> matchedFaqList) {
        if (!isVisible()) return;
        // 如果 matchedFaqList 为 null，应该 restoreViewState 而不是 showEmpty
        if (matchedFaqList != null && matchedFaqList.size() == 0) {
            showSearchEmpty();
        } else {
            restoreViewState();
        }
        mSearchAdapter.update(matchedFaqList);
    }

    @Override
    protected int getLoadingTargetViewId() {
        return ResResolver.getViewId("aihelp_rv_faq_search");
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (debounceTimer != null) {
            debounceTimer.cancel();
        }
        EventTracker.INSTANCE.log(EventType.HC_PAGE_SEARCH_CLOSED);
    }
}
