package net.aihelp.ui.task.detail;

import android.app.Activity;
import android.content.Intent;
import android.graphics.Color;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;
import android.widget.Toast;

import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.ui.BaseFragment;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.permission.AIHelpPermissions;
import net.aihelp.data.attachment.AttachmentPicker;
import net.aihelp.data.event.SearchViewVisibilityChangeEvent;
import net.aihelp.data.event.UpdateTitleEvent;
import net.aihelp.data.logic.TaskDetailPresenter;
import net.aihelp.data.model.task.ReplyMessage;
import net.aihelp.data.model.task.TaskDetailEntity;
import net.aihelp.ui.adapter.task.detail.TaskReplyListAdapter;
import net.aihelp.ui.adapter.task.widget.EvaluateTicketView;
import net.aihelp.ui.task.helper.EvaluateTicketHelper;
import net.aihelp.ui.widget.AIHelpButton;
import net.aihelp.ui.widget.StepView;
import net.aihelp.utils.DateFormatUtil;
import net.aihelp.utils.DebounceHelper;
import net.aihelp.utils.FastClickValidator;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.RichTextSlicer;
import net.aihelp.utils.Styles;
import net.aihelp.utils.TLog;
import net.aihelp.utils.UploadFileHelper;

import org.json.JSONArray;

import java.util.List;

import androidx.annotation.NonNull;
import androidx.appcompat.widget.AppCompatButton;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class TaskDetailFragment extends BaseFragment<TaskDetailPresenter> implements View.OnClickListener,
        TaskReplyListAdapter.OnReplyEventListener, TaskReplyComposer.OnUserReplyListener, EvaluateTicketView.OnAIHelpEvaluateViewListener {

    private TaskReplyComposer replyComposer;

    private StepView stepView;
    private EvaluateTicketView evaluateView;
    private TaskReplyListAdapter replyListAdapter;
    private TextView tvTaskTime;
    private LinearLayout llTaskContent;
    private AIHelpButton btnReply;
    private LinearLayout llEvaluate;

    private String taskCode;

    public static TaskDetailFragment newInstance(Bundle bundle) {
        TaskDetailFragment faqFragment = new TaskDetailFragment();
        faqFragment.setArguments(bundle);
        return faqFragment;
    }

    @Override
    protected void getBundleBeforeDataPrepared(Bundle extras) {
        EventBus.getDefault().post(new SearchViewVisibilityChangeEvent(false));
        taskCode = extras.getString(IntentValues.TASK_CODE, "");
    }

    @Override
    public void onResume() {
        super.onResume();
        EventBus.getDefault().post(new UpdateTitleEvent(UpdateTitleEvent.CUSTOM_SERVICE, null,
                ResResolver.getString("aihelp_my_task")));
    }

    @Override
    protected void initEventAndData(View view) {
        replyComposer = new TaskReplyComposer();

        RelativeLayout rlStatusContainer = get("aihelp_rl_status");
        TextView tvStatusTitle = get("aihelp_tv_status_title");
        stepView = get("aihelp_step_view");

        evaluateView = get("aihelp_v_evaluate");
        evaluateView.setOnAIHelpEvaluateViewListener(this);

        RelativeLayout rlReplyContainer = get("aihelp_rl_reply");
        TextView tvReplyTitle = get("aihelp_tv_reply_title");
        RecyclerView rvReply = get("aihelp_rv_reply");
        btnReply = get("aihelp_btn_reply");

        RelativeLayout rlTaskContainer = get("aihelp_rl_task");
        TextView tvTaskTitle = get("aihelp_tv_task_title");
        tvTaskTime = get("aihelp_tv_task_time");
        llTaskContent = get("aihelp_ll_task_content");

        int colorFromHex = Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha);
        rlStatusContainer.setBackground(Styles.getDrawable(colorFromHex, 0));
        rlReplyContainer.setBackground(Styles.getDrawable(colorFromHex, 0));
        rlTaskContainer.setBackground(Styles.getDrawable(colorFromHex, 0));

        int textColor = Color.parseColor(CustomConfig.CommonSetting.textColor);
        Styles.reRenderTextView(tvStatusTitle, ResResolver.getString("aihelp_task_status"), textColor);
        Styles.reRenderTextView(tvReplyTitle, ResResolver.getString("aihelp_reply_content"), textColor);
        Styles.reRenderTextView(tvTaskTitle, ResResolver.getString("aihelp_task_content"), textColor);

        LinearLayoutManager linearLayoutManager = new LinearLayoutManager(getContext()) {
            @Override
            public boolean canScrollVertically() {
                return false;
            }
        };
        linearLayoutManager.setReverseLayout(true);
        rvReply.setLayoutManager(linearLayoutManager);
        replyListAdapter = new TaskReplyListAdapter(getContext(), this);
        replyListAdapter.setOnReplyEventListener(this);
        rvReply.setAdapter(replyListAdapter);

        btnReply.useSmallSize().setText(ResResolver.getString("aihelp_reply"));
        btnReply.setOnClickListener(this);

        llEvaluate = get("aihelp_ll_rate");
        llEvaluate.setBackgroundColor(Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha));
        TextView rateInvite = get("aihelp_tv_invite_rate");
        Styles.reRenderTextView(rateInvite, CustomConfig.InitSetting.csInviteEvaluate);

        AppCompatButton rateButton = get("aihelp_btn_go_rate");
        rateButton.setText(ResResolver.getString("aihelp_rate_button"));
        rateButton.setBackground(Styles.getDrawable(Color.parseColor(CustomConfig.CommonSetting.interactElementTextColor), 8));
        rateButton.setOnClickListener(this);

        mPresenter.requestTaskContent(taskCode);
    }

    @Override
    protected int getLayout() {
        return ResResolver.getLayoutId("aihelp_fra_task_detail");
    }

    public void refreshTaskContent(TaskDetailEntity entity, List<ReplyMessage> replyList) {
        stepView.update(entity.getStayTime(), entity.getStatusList(),
                entity.getStatus(), CustomConfig.CommonSetting.interactElementTextColor
        );
        btnReply.setVisibility(stepView.isFinished() ? View.GONE : View.VISIBLE);

        evaluateView.setVisibility(entity.isShowFeedback() ? View.VISIBLE : View.GONE);
        evaluateView.setupView(entity.isViolationSla(), entity.getBadFeedbackNum(), entity.getGoodFeedbackNum());

        replyListAdapter.update(replyList);

        Styles.reRenderTextView(tvTaskTime, DateFormatUtil.dateFormat(entity.getCreateTime(), null));
        llTaskContent.addView(RichTextSlicer.getRichTextView(getContext(), this, entity.getContent()));

        if (entity.isTaskFinished()) {
            updateRatingStatus(entity.isEvaluate());
        }
    }

    @Override
    public void onRetrySendingMessage(int position, ReplyMessage replyMessage) {
        // Only remove item when we're resending the message after a failed sending
        // If the resending succeed, the reply message will be updated to NORMAL status,
        // and the following logic will update the message
        if (replyMessage.getMsgStatus() == ReplyMessage.STATUS_SENDING) {
            replyListAdapter.remove(position);
        }
        onUserReply(replyMessage);
    }

    @Override
    public void onClick(View v) {
        if (FastClickValidator.validate(0.5f)) {
            if (v.getId() == ResResolver.getViewId("aihelp_btn_reply")) {
                replyComposer.showReplyComposer(this, getContext());
                replyComposer.setOnUserReplyListener(this);
            }
        }
        if (v.getId() == ResResolver.getViewId("aihelp_btn_go_rate")) {
            EvaluateTicketHelper.getInstance().showRateSupport(getContext(), new EvaluateTicketHelper.OnConfirmEvaluateListener() {
                @Override
                public boolean onConfirmEvaluate(int rate, String selectedIds, String content) {
                    ReplyMessage evaluateMsg = mPresenter.buildEvaluationMessage(rate, content, System.currentTimeMillis());
                    replyListAdapter.update(evaluateMsg);
                    mPresenter.onUserEvaluate(taskCode, rate, selectedIds, content);
                    return true;
                }
            });
        }
    }

    @Override
    public void onUserReply(ReplyMessage replyMessage) {
        if (replyMessage.getMsgType() == ReplyMessage.MSG_TYPE_IMAGE
                || replyMessage.getMsgType() == ReplyMessage.MSG_TYPE_VIDEO
                || replyMessage.getMsgType() == ReplyMessage.MSG_TYPE_FILE) {
            if (replyMessage.getMsgStatus() == ReplyMessage.STATUS_SENDING) {
                replyListAdapter.update(replyMessage);
            } else {
                for (int i = replyListAdapter.getDataList().size() - 1; i >= 0; i--) {
                    ReplyMessage msg = replyListAdapter.getDataList().get(i);
                    if (msg.getTimeStamp() == replyMessage.getTimeStamp()) {
                        msg.setMsgStatus(replyMessage.getMsgStatus() == ReplyMessage.STATUS_NORMAL
                                ? ReplyMessage.STATUS_NORMAL : ReplyMessage.STATUS_FAILURE);
                        replyListAdapter.notifyItemChanged(i);
                        break;
                    }
                }
            }
        } else {
            replyListAdapter.update(replyMessage);
        }

        if (replyMessage.getMsgStatus() == ReplyMessage.STATUS_NORMAL) {
            mPresenter.onUserReply(taskCode, replyMessage.getContent(), replyMessage.getReplyFormat());
        }
    }

    public void updateRatingStatus(boolean isEvaluated) {
        llEvaluate.setVisibility(isEvaluated ? View.GONE : View.VISIBLE);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        UploadFileHelper.INSTANCE.setOnUploadFileListener(null);
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        AIHelpPermissions.getInstance().onRequestPermissionsResult(permissions, grantResults);
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent intent) {
        super.onActivityResult(requestCode, resultCode, intent);
        AttachmentPicker.INSTANCE.onAttachmentRequestResult(requestCode, resultCode, intent);
    }

    @Override
    public void onEvaluated(boolean isHelpful, int count) {
        DebounceHelper.INSTANCE.debounce(new Runnable() {
            @Override
            public void run() {
                mPresenter.onUserFeedback(taskCode, isHelpful, count);
            }
        }, 500);
    }

}
