package net.aihelp.ui.widget;

import android.content.Context;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.TextView;

import net.aihelp.common.API;
import net.aihelp.common.CustomConfig;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.data.model.rpa.msg.BotMessage;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import org.json.JSONObject;

import java.util.Iterator;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

public class AIHelpEvaluateButtonView extends FrameLayout implements View.OnClickListener {

    public static final int STATE_NORMAL = 1;
    public static final int STATE_INVISIBLE = 2;
    public static final int STATE_HELPFUL = 3;
    public static final int STATE_UNHELPFUL = 4;

    private final AIHelpButton mBtnUnHelpful;
    private final AIHelpButton mBtnHelpful;
    private final TextView mTvShowThanks;

    private int minWidth;
    private int maxWidth;
    private OnAIHelpEvaluateViewCallback listener;

    public int getMinWidth() {
        return minWidth;
    }

    private void resetEvaluateState() {
        setVisibility(VISIBLE);
        mBtnHelpful.setVisibility(View.VISIBLE);
        mBtnUnHelpful.setVisibility(View.VISIBLE);
        mTvShowThanks.setVisibility(View.GONE);
    }

    private void setEvaluateState(int evaluateState) {
        resetEvaluateState();
        switch (evaluateState) {
            case STATE_NORMAL:
                int buttonWidth = Math.max(mBtnUnHelpful.getMeasuredWidth(), mBtnHelpful.getMeasuredWidth());
                minWidth = buttonWidth * 2 + Styles.dpToPx(getContext(), 20);
                if (maxWidth > 0 && buttonWidth * 2 + Styles.dpToPx(getContext(), 20) > maxWidth) {
                    buttonWidth = (maxWidth - Styles.dpToPx(getContext(), 20)) / 2;
                }
                ViewGroup.LayoutParams unHelpfulParams = mBtnUnHelpful.getLayoutParams();
                unHelpfulParams.width = buttonWidth;
                mBtnUnHelpful.setLayoutParams(unHelpfulParams);
                ViewGroup.LayoutParams helpfulParams = mBtnHelpful.getLayoutParams();
                helpfulParams.width = buttonWidth;
                mBtnHelpful.setLayoutParams(helpfulParams);
                break;
            case STATE_INVISIBLE:
                setVisibility(GONE);
                break;
            case STATE_HELPFUL:
            case STATE_UNHELPFUL:
                mBtnHelpful.setVisibility(View.GONE);
                mBtnUnHelpful.setVisibility(View.GONE);
                mTvShowThanks.setVisibility(View.VISIBLE);
                if (evaluateState == STATE_HELPFUL) {
                    Styles.reRenderTextView(mTvShowThanks, CustomConfig.InitSetting.commonPositiveFeedbackHint, 0.5f);
                } else {
                    Styles.reRenderTextView(mTvShowThanks, CustomConfig.InitSetting.commonNegativeFeedbackHint, 0.5f);
                }
                mTvShowThanks.measure(0, 0);
                minWidth = mTvShowThanks.getMeasuredWidth();
                break;
        }
    }

    public void setOnAIHelpEvaluateViewCallback(OnAIHelpEvaluateViewCallback listener) {
        this.listener = listener;
    }

    public AIHelpEvaluateButtonView(@NonNull Context context) {
        this(context, null);
    }

    public AIHelpEvaluateButtonView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public AIHelpEvaluateButtonView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);

        // 加载 xml 布局
        View rootView = View.inflate(context, ResResolver.getLayoutId("aihelp_evaluate_button_view"), this);

        // 点踩视图
        mBtnUnHelpful = rootView.findViewById(ResResolver.getViewId("aihelp_btn_un_helpful"));
        mBtnUnHelpful.setText(ResResolver.getString("aihelp_faq_unhelpful"));
        mBtnUnHelpful.measure(0, 0);

        // 点赞视图
        mBtnHelpful = rootView.findViewById(ResResolver.getViewId("aihelp_btn_helpful"));
        mBtnHelpful.setText(ResResolver.getString("aihelp_faq_helpful"));
        mBtnHelpful.measure(0, 0);

        // 评价后视图
        mTvShowThanks = rootView.findViewById(ResResolver.getViewId("aihelp_tv_thanks"));

        // 给点赞、点踩和提建议设置点击事件
        mBtnUnHelpful.setOnClickListener(this);
        mBtnHelpful.setOnClickListener(this);
    }

    /**
     * 根据开关以及 userFeedback 的值决定显示点赞点踩还是结果提示
     */
    public void refreshViewState(int userFeedback) {
        if (CustomConfig.InitSetting.isEvaluationForBotEnable) {
            switch (userFeedback) {
                case BotMessage.USER_FEEDBACK_INVISIBLE:
                    setEvaluateState(STATE_INVISIBLE);
                    break;
                case BotMessage.USER_FEEDBACK_NONE:
                    setEvaluateState(STATE_NORMAL);
                    break;
                case BotMessage.USER_FEEDBACK_POSITIVE:
                    setEvaluateState(STATE_HELPFUL);
                    break;
                case BotMessage.USER_FEEDBACK_NEGATIVE:
                    setEvaluateState(STATE_UNHELPFUL);
                    break;
            }
        } else {
            setEvaluateState(STATE_INVISIBLE);
        }
    }

    @Override
    public void onClick(View v) {
        if (v.getId() == ResResolver.getViewId("aihelp_btn_un_helpful")) {
            setEvaluateState(STATE_UNHELPFUL);
            postHelpfulStatusForAnswerBot(false);
        }

        if (v.getId() == ResResolver.getViewId("aihelp_btn_helpful")) {
            setEvaluateState(STATE_HELPFUL);
            postHelpfulStatusForAnswerBot(true);
        }

    }

    private void postHelpfulStatusForAnswerBot(boolean helpful) {
        if (listener != null) listener.onEvaluated(helpful);
        try {
            JSONObject params = new JSONObject();
            params.put("type", helpful ? "like" : "unlike");
            if (listener != null && listener.requestDataForFeedback() != null) {
                JSONObject dataForFeedback = listener.requestDataForFeedback();
                Iterator<String> keys = dataForFeedback.keys();
                while (keys.hasNext()) {
                    String key = keys.next();
                    params.put(key, dataForFeedback.opt(key));
                }
            }
            AIHelpRequest.getInstance().requestPostByJson(API.EVALUATE_ANSWER_BOT_FAQ, params, null);
        } catch (Exception e) {
            // ignored
        }
    }

    public void setMaxWidth(int maxWidth) {
        this.maxWidth = maxWidth;
    }

    public static abstract class OnAIHelpEvaluateViewCallback {
        public void onEvaluated(boolean isHelpful) {

        }

        public JSONObject requestDataForFeedback() {
            return null;
        }
    }

}
