package net.aihelp.ui.widget;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Picture;
import android.os.Build;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;
import android.widget.TextView;

import net.aihelp.R;
import net.aihelp.common.CustomConfig;
import net.aihelp.utils.DateFormatUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import java.util.ArrayList;
import java.util.List;

import androidx.core.graphics.drawable.DrawableCompat;
import androidx.core.view.ViewCompat;
import androidx.vectordrawable.graphics.drawable.VectorDrawableCompat;

public class StepView extends View {

    private Paint bgPaint;
    private Paint progressPaint;

    private final float radius = 12;

    private String backgroundColor = "#d8d8d8";
    private String progressColor = "#ffffff";
    private int textSize = 14;

    private float startX;
    private float stopX;
    private float bgCenterY;

    private int currentStep = 0;

    private int textPadding = 10;
    private int interval;

    private long duration;

    private List<String> steps = new ArrayList<>();

    public boolean isFinished() {
        return currentStep >= steps.size() - 1;
    }

    public void setBackgroundColor(String backgroundColor) {
        this.backgroundColor = backgroundColor;
    }

    public void update(long duration, List<String> steps, String currentStep, String progressColor) {
        if (steps != null && steps.size() > 0) {
            this.duration = duration;
            this.steps = steps;
            this.currentStep = steps.indexOf(currentStep);
            this.progressColor = progressColor;
            progressPaint.setColor(Color.parseColor(progressColor));
            forceLayout();
            invalidate();
        }
    }

    public StepView(Context context) {
        this(context, null);
    }

    public StepView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public StepView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    private void init() {
        textSize = dpToPx(textSize);
        textPadding = dpToPx(textPadding);

        bgPaint = new Paint();
        bgPaint.setAntiAlias(true);
        bgPaint.setStyle(Paint.Style.FILL);
        bgPaint.setColor(Color.parseColor(backgroundColor));
        bgPaint.setStrokeWidth(2f);
        bgPaint.setTextSize(textSize);
        bgPaint.setTextAlign(Paint.Align.CENTER);

        progressPaint = new Paint();
        progressPaint.setAntiAlias(true);
        progressPaint.setStyle(Paint.Style.FILL);
        progressPaint.setColor(Color.parseColor(progressColor));
        progressPaint.setStrokeWidth(2f);
        progressPaint.setTextSize(textSize);
        progressPaint.setTextAlign(Paint.Align.CENTER);
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        // the heightMeasureSpec for ScrollView is UNSPECIFIED, need to be corrected for the view to be displayed
        if (heightMeasureSpec == 0) {
            heightMeasureSpec = MeasureSpec.makeMeasureSpec(widthMeasureSpec, MeasureSpec.AT_MOST);
        }
        int widthMode = MeasureSpec.getMode(widthMeasureSpec);
        int heightMode = MeasureSpec.getMode(heightMeasureSpec);
        int bgWidth;
        if (widthMode == MeasureSpec.EXACTLY) {
            bgWidth = MeasureSpec.getSize(widthMeasureSpec);
        } else {
            bgWidth = getPaddingLeft() + dpToPx(300) + getPaddingRight();
        }

        int bgHeight;
        if (heightMode == MeasureSpec.EXACTLY) {
            bgHeight = MeasureSpec.getSize(heightMeasureSpec) - getPaddingTop() - getPaddingBottom();
        } else {
            bgHeight = (int) (textSize * 2 + radius * 2 + textPadding + textSize * 2) + getPaddingTop() + getPaddingBottom();
        }

        startX = getPaddingLeft() + radius;
        stopX = bgWidth - radius - getPaddingRight();

        if (Styles.isLayoutRtl(this)) {
            startX = bgWidth - getPaddingLeft() - radius;
            stopX = radius + getPaddingRight();
        }

        bgCenterY = bgHeight * 1.0f / 2;

        interval = (int) ((stopX - startX) / (steps.size() - 1));

        super.onMeasure(MeasureSpec.makeMeasureSpec(bgWidth, widthMode), MeasureSpec.makeMeasureSpec(bgHeight, heightMode));
    }

    @Override
    protected void onDraw(Canvas canvas) {
        drawBg(canvas);
        drawProgress(canvas);
        drawText(canvas);
        drawTip(canvas);
    }

    public float getBaseLine(float y, Paint mPaint) {
        Paint.FontMetrics fontMetrics = mPaint.getFontMetrics();
        float distance = (fontMetrics.bottom - fontMetrics.top) / 2 - fontMetrics.bottom;
        return y + distance;
    }

    private void drawText(Canvas canvas) {
        for (int i = 0; i <= steps.size() - 1; i++) {
            int y = (int) (i % 2 == 0 ? (getBaseLine(bgCenterY, progressPaint) - textSize) : (getBaseLine(bgCenterY, progressPaint) + textSize));
            if (i <= currentStep) {
                float x = getDrawTextStartPosition(i, progressPaint);
                canvas.drawText(steps.get(i), x, y, progressPaint);
            } else {
                float x = getDrawTextStartPosition(i, bgPaint);
                canvas.drawText(steps.get(i), x, y, bgPaint);
            }
        }
    }

    private void drawTip(Canvas canvas) {
        for (int i = 0; i <= steps.size() - 1; i++) {
            int y = (int) (i % 2 == 0 ? (getBaseLine(bgCenterY, progressPaint) - textSize * 2) :
                    (getBaseLine(bgCenterY, progressPaint) + textSize * 2));
            if (currentStep < steps.size() - 1 && i == currentStep) {
                String text = DateFormatUtil.getFormattedMilliSeconds(duration);
                if (!TextUtils.isEmpty(text)) {
                    float x = getDrawTextStartPosition(i, progressPaint);
                    float measuredWidth = progressPaint.measureText(text);
                    float measuredHeight = progressPaint.getTextSize();
                    int drawableSize = 0;

                    VectorDrawableCompat drawable = VectorDrawableCompat.create(getResources(),
                            ResResolver.getDrawableId("aihelp_svg_ic_time"), null);
                    if (drawable != null) {
                        DrawableCompat.setTint(DrawableCompat.wrap(drawable).mutate(),
                                Styles.getColor(CustomConfig.CommonSetting.interactElementTextColor));
                        drawableSize = dpToPx(12);
                        canvas.save();
                        // 移动画布到绘制的位置，默认从 drawable 的左上角开始绘制，所以需要往文字前面挪一挪
                        float offsetX = i == 0 ? 0 : (drawableSize / 2.0f + measuredWidth / 2 + 5);
                        canvas.translate(x - offsetX, y - drawableSize / 2.0f - measuredHeight / 2 + 5);
                        drawable.setBounds(0, 0, drawableSize, drawableSize);
                        drawable.draw(canvas);
                        canvas.restore();
                    }
                    float textOffsetX = i == 0 ? drawableSize : (drawableSize / 2.0f);
                    canvas.drawText(text, x + textOffsetX, y, progressPaint);
                }
            }
        }
    }

    private float getDrawTextStartPosition(int index, Paint paint) {
        float x = startX + (index * interval);
        if (index == 0) {
            if (Styles.isLayoutRtl(this)) {
                paint.setTextAlign(Paint.Align.RIGHT);
                x = x + radius;
            } else {
                paint.setTextAlign(Paint.Align.LEFT);
                x = x - radius;
            }
        } else if (index == steps.size() - 1) {
            if (Styles.isLayoutRtl(this)) {
                paint.setTextAlign(Paint.Align.LEFT);
                x = x - radius;
            } else {
                paint.setTextAlign(Paint.Align.RIGHT);
                x = x + radius;
            }
        } else {
            paint.setTextAlign(Paint.Align.CENTER);
        }
        return x;
    }

    private void drawProgress(Canvas canvas) {
        int linePro;
        float lastLeft = startX;
        for (int i = 0; i <= currentStep; i++) {
            if (i == 0) {
                linePro = 0;
            } else {
                linePro = interval;
            }
            canvas.drawLine(lastLeft, bgCenterY, lastLeft + linePro, bgCenterY, progressPaint);
            lastLeft = lastLeft + linePro;
            canvas.drawCircle(startX + (i * interval), bgCenterY, radius, progressPaint);
        }
    }

    private void drawBg(Canvas canvas) {
        canvas.drawLine(startX, bgCenterY, stopX, bgCenterY, bgPaint);
        for (int i = 0; i <= steps.size() - 1; i++) {
            canvas.drawCircle(startX + (i * interval), bgCenterY, radius, bgPaint);
        }
    }

    public int dpToPx(float dp) {
        return (int) (dp * getContext().getResources().getDisplayMetrics().density);
    }

    private boolean isLayoutRtl() {
        return ViewCompat.getLayoutDirection(this) == ViewCompat.LAYOUT_DIRECTION_RTL;
    }

}
