package net.aihelp.utils;

import android.app.ActivityManager;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.text.format.Formatter;
import android.util.Log;

import net.aihelp.BuildConfig;
import net.aihelp.common.Const;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.UserProfile;
import net.aihelp.config.AIHelpContext;
import net.aihelp.config.enums.PushPlatform;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.data.model.init.PrivacyControlEntity;

import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;

import androidx.annotation.RestrictTo;

@RestrictTo(RestrictTo.Scope.LIBRARY)
public class DeviceInfoUtil {

    private final Context context;

    private DeviceInfoUtil(Context context) {
        this.context = context;
    }

    private static DeviceInfoUtil sInstance;

    public static DeviceInfoUtil getInstance() {
        if (sInstance == null) {
            Context context = AIHelpContext.getInstance().getContext();
            sInstance = new DeviceInfoUtil(context.getApplicationContext());
        }
        return sInstance;
    }

    public int getStatusBarHeight() {
        int statusBarHeight = 0;
        //获取status_bar_height资源的ID
        int resourceId = context.getResources().getIdentifier("status_bar_height", "dimen", "android");
        if (resourceId > 0) {
            //根据资源ID获取响应的尺寸值
            statusBarHeight = context.getResources().getDimensionPixelSize(resourceId);
        }
        return statusBarHeight;
    }

    public String getSimCountryIso() {
        TelephonyManager tm = (TelephonyManager) this.context.getSystemService(Context.TELEPHONY_SERVICE);
        if (tm != null) {
            String simCountryIso = tm.getSimCountryIso();
            if (!TextUtils.isEmpty(simCountryIso)) {
                return simCountryIso.toUpperCase();
            }
        }
        return Locale.getDefault().getCountry();
    }

    public String getTimeStamp() {
        return (new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", Locale.ENGLISH)).format(new Date());
    }

    /**
     * 获取android当前可用运行内存大小
     */
    public String getAvailMemory() {
        ActivityManager am = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
        ActivityManager.MemoryInfo mi = new ActivityManager.MemoryInfo();
        am.getMemoryInfo(mi);
        // mi.availMem; 当前系统的可用内存
        return Formatter.formatFileSize(context, mi.availMem);// 将获取的内存大小规格化
    }

    /**
     * 获取android总运行内存大小
     */
    public String getTotalMemory() {
        String str1 = "/proc/meminfo";// 系统内存信息文件
        String str2;
        String[] arrayOfString;
        long initial_memory = 0;
        try {
            FileReader localFileReader = new FileReader(str1);
            BufferedReader localBufferedReader = new BufferedReader(localFileReader, 8192);
            str2 = localBufferedReader.readLine();// 读取meminfo第一行，系统总内存大小
            arrayOfString = str2.split("\\s+");
            for (String num : arrayOfString) {
                Log.i(str2, num + "\t");
            }
            // 获得系统总内存，单位是KB
            int i = Integer.parseInt(arrayOfString[1]);
            //int值乘以1024转换为long类型
            initial_memory = (long) i * 1024;
            localBufferedReader.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return Formatter.formatFileSize(context, initial_memory);// Byte转换为KB或者MB，内存大小规格化
    }

    public String getCarrierName() {
        TelephonyManager tm = (TelephonyManager) this.context.getSystemService(Context.TELEPHONY_SERVICE);
        return tm == null ? "" : tm.getNetworkOperatorName();
    }

    public String getNetworkType() {
        String type = "Unknown";
        try {
            ConnectivityManager cm = (ConnectivityManager) this.context.getSystemService(Context.CONNECTIVITY_SERVICE);
            if (cm != null) {
                NetworkInfo ani = cm.getActiveNetworkInfo();
                if (ani != null) {
                    type = ani.getTypeName();
                }
            }
        } catch (SecurityException var4) {
            type = null;
        }
        return type;
    }

    public String getBatteryStatus() {
        IntentFilter filter = new IntentFilter("android.intent.action.BATTERY_CHANGED");
        Intent batteryStatus = this.context.registerReceiver(null, filter);
        if (batteryStatus == null) {
            return "Not charging";
        } else {
            int status = batteryStatus.getIntExtra("status", -1);
            boolean isCharging = status == 2 || status == 5;
            return isCharging ? "Charging" : "Not charging";
        }
    }

    public String getBatteryLevel() {
        IntentFilter filter = new IntentFilter("android.intent.action.BATTERY_CHANGED");
        Intent batteryStatus = this.context.registerReceiver(null, filter);
        if (batteryStatus == null) {
            return "";
        } else {
            int level = batteryStatus.getIntExtra("level", -1);
            int scale = batteryStatus.getIntExtra("scale", -1);
            int batteryPct = (int) ((float) level / (float) scale * 100.0F);
            return batteryPct + "%";
        }
    }

    public String getTotalDiskSpace() {
        double bytesInOneGB = 1073741824.0D;
        StatFs phoneStat = new StatFs(Environment.getDataDirectory().getPath());
        double total_phone_memory;
        if (Build.VERSION.SDK_INT >= 18) {
            total_phone_memory = (double) phoneStat.getBlockCountLong() * (double) phoneStat.getBlockSizeLong() / bytesInOneGB;
            total_phone_memory = (double) Math.round(total_phone_memory * 100.0D) / 100.0D;
        } else {
            total_phone_memory = (double) phoneStat.getBlockCount() * (double) phoneStat.getBlockSize() / bytesInOneGB;
            total_phone_memory = (double) Math.round(total_phone_memory * 100.0D) / 100.0D;
        }
        return total_phone_memory + "GB";
    }

    public String getRemainDiskSpace() {
        double bytesInOneGB = 1073741824.0D;
        StatFs phoneStat = new StatFs(Environment.getDataDirectory().getPath());
        double free_phone_memory;
        if (Build.VERSION.SDK_INT >= 18) {
            free_phone_memory = (double) phoneStat.getAvailableBlocksLong() * (double) phoneStat.getBlockSizeLong() / bytesInOneGB;
            free_phone_memory = (double) Math.round(free_phone_memory * 100.0D) / 100.0D;
        } else {
            free_phone_memory = (double) phoneStat.getAvailableBlocks() * (double) phoneStat.getBlockSize() / bytesInOneGB;
            free_phone_memory = (double) Math.round(free_phone_memory * 100.0D) / 100.0D;
        }
        return free_phone_memory + "GB";
    }

    private static JSONObject getHostGameInfo(PrivacyControlEntity entity) {
        String hostPackageName = "unknown";
        String hostAppVersion = "0.0.0";
        String hostApplicationName = "unknown";
        try {
            Context context = AIHelpContext.getInstance().getContext();
            hostPackageName = context.getPackageName();
            ApplicationInfo ai = context.getPackageManager().getApplicationInfo(hostPackageName, 0);
            hostAppVersion = context.getPackageManager().getPackageInfo(hostPackageName, 0).versionName;
            hostApplicationName = (String) context.getPackageManager().getApplicationLabel(ai);
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        JSONObject params = new JSONObject();
        try {
            if (entity != null && entity.getApplicationIdentifier()) {
                params.put("Application_Identifier", hostPackageName);
            }
            if (entity != null && entity.getApplicationVersion()) {
                params.put("Application_Version", hostAppVersion);
            }
            if (entity != null && entity.getApplicationName()) {
                params.put("Name", hostApplicationName);
            }
            if (entity != null && entity.getServerId()) {
                params.put("ServerId", UserProfile.SERVER_ID);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return params;
    }

    private JSONObject getHardwareInfo(PrivacyControlEntity entity) {
        JSONObject hardwareInfo = new JSONObject();
        try {
            hardwareInfo.put("DeviceId", DeviceUuidFactory.id(context));
            if (entity != null && entity.getTotalMemory()) {
                hardwareInfo.put("totalMemory", getTotalMemory());
            }
            if (entity != null && entity.getAvailableMemory()) {
                hardwareInfo.put("availableMemory", getAvailMemory());
            }
            if (entity != null && entity.getDeviceModel()) {
                hardwareInfo.put("Device_Model", String.format("%s %s", Build.MANUFACTURER, Build.MODEL));
            }
            if (entity != null && entity.getFreeSpacePhone()) {
                hardwareInfo.put("Free_Space", getRemainDiskSpace());
            }
            if (entity != null && entity.getTotalSpacePhone()) {
                hardwareInfo.put("Total_Space", getTotalDiskSpace());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return hardwareInfo;
    }

    private JSONObject getOtherInfo(PrivacyControlEntity entity) {
        JSONObject otherInfo = new JSONObject();
        try {
            otherInfo.put("Platform", "android");
            otherInfo.put("Language", Const.ORIGINAL_LANGUAGE);
            otherInfo.put("SDK_Version", BuildConfig.SDK_VERSION);
            if (entity != null && entity.getOsVersion()) {
                otherInfo.put("OS_Version", Build.VERSION.RELEASE);
            }
            if (entity != null && entity.getNetworkType()) {
                otherInfo.put("Network_Type", getNetworkType());
            }
            if (entity != null && entity.getOperator()) {
                otherInfo.put("Carrier", getCarrierName());
            }
            if (entity != null && entity.getCountryCode()) {
                otherInfo.put("Country_Code", getSimCountryIso());
            }
            if (!TextUtils.isEmpty(Const.PUSH_INFO) && Const.PUSH_INFO.contains("|")) {
                String pushToken = Const.PUSH_INFO.split("\\|")[0];
                int pushPlatform = Integer.parseInt(Const.PUSH_INFO.split("\\|")[1]);
                JSONObject pushInfo = JsonHelper.getJsonObject();
                pushInfo.put("Token", pushToken);
                pushInfo.put("PushTypeId", pushPlatform);
                otherInfo.put("PushToken", pushInfo);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return otherInfo;
    }

    private String fillStoryNodeToCustomData() {

        if (TextUtils.isEmpty(UserProfile.ENTRANCE_TAGS)) {
            return UserProfile.CUSTOM_DATA;
        }

        try {
            JSONObject customData = new JSONObject(UserProfile.CUSTOM_DATA);

            JSONObject elvaMetaData = JsonHelper.getJsonObject(customData, "elva-custom-metadata");
            String originUserTags = JsonHelper.optString(elvaMetaData, "hs-tags");

            if (TextUtils.isEmpty(originUserTags)) {
                insertUserTagsIntoCustomData(elvaMetaData, UserProfile.ENTRANCE_TAGS);
            } else {
                insertUserTagsIntoCustomData(elvaMetaData, originUserTags + "," + UserProfile.ENTRANCE_TAGS);
            }

            JSONObject returnData = new JSONObject();
            returnData.put("elva-custom-metadata", elvaMetaData);
            returnData.put("hs-custom-metadata", new JSONObject(elvaMetaData.toString()));

            return returnData.toString();
        } catch (Exception e) {
            e.printStackTrace();
            return UserProfile.CUSTOM_DATA;
        }
    }

    private void insertUserTagsIntoCustomData(JSONObject metaData, String userTags) {
        String[] userTagsArray = userTags.split(",");
        if (userTagsArray.length > 0) {
            try {
                metaData.put("elva-tags", Arrays.asList(userTagsArray));
                metaData.put("hs-tags", userTags);
            } catch (Exception e) {
                // ignored
            }
        }
    }

    public String getProxyConfiguration() {
        String host = System.getProperty("http.proxyHost");
        String port = System.getProperty("http.proxyPort");
        if (!TextUtils.isEmpty(host) || !TextUtils.isEmpty(port)) {
            return host + ":" + port;
        } else {
            return "";
        }
    }

    public JSONObject getGameInfo() {
        JSONObject gameInfo = new JSONObject();
        try {
            PrivacyControlEntity entity = CustomConfig.InitSetting.privacyControlData;
            gameInfo.put("APPLICATION", getHostGameInfo(entity));
            gameInfo.put("HARDWARE", getHardwareInfo(entity));
            gameInfo.put("OTHER", getOtherInfo(entity));
            gameInfo.put("CUSTOMDATA", fillStoryNodeToCustomData());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return gameInfo;
    }

    public String getDeviceIdFromCustomData() {
        try {
            JSONObject customData = new JSONObject(UserProfile.CUSTOM_DATA);
            JSONObject elvaMetaData = JsonHelper.getJsonObject(customData, "elva-custom-metadata");
            return elvaMetaData.optString("deviceId");
        } catch (Exception e) {
            return "";
        }
    }

}

