package net.aihelp.utils;

import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.provider.MediaStore;
import android.text.TextUtils;
import android.webkit.MimeTypeMap;

import net.aihelp.config.AIHelpContext;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.OutputStream;

public class DownloadHelper {

    public interface OnDownloadProgressChangedListener {
        void onProgressChanged(int progress);
    }

    public static void save(Context context, String path, OnDownloadProgressChangedListener listener) {
        String fileName = getSplitFileName(path);
        String savePath = getTargetSavePath(path, fileName);
        AIHelpRequest.getInstance().requestDownloadFile(path, savePath, new ReqCallback<String>() {

            @Override
            public void onReqProgress(long totalContentLength, long currentTransferred, int progress) {
                if (listener != null) {
                    listener.onProgressChanged(progress);
                }
            }

            @Override
            public void onReqSuccess(String result) {
                try {
                    if (RegexUtil.isVideoFile(savePath)) {
                        insertVideo(context, savePath);
                    } else if (RegexUtil.isGifFile(savePath)) {
                        insertGif(context, savePath);
                    } else {
                        insertImage(context, savePath, fileName);
                    }
                    if (!RegexUtil.isVideoFile(savePath)) {
                        ToastUtil.INSTANCE.makeTextWithIcon(context, ResResolver.getString("aihelp_save_seccessfully"), false);
                    }
                } catch (Exception e) {
                    String saveFailedToastMessage = String.format("%s: %s", ResResolver.getString("aihelp_save_seccessfully"), savePath);
                    ToastUtil.INSTANCE.makeRawToast(context, saveFailedToastMessage);
                }
            }
        });
    }

    private static void insertVideo(Context context, String filePath) {
        ContentResolver resolver = context.getContentResolver();
        Uri collectionUri;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            collectionUri = MediaStore.Video.Media.getContentUri(MediaStore.VOLUME_EXTERNAL_PRIMARY);
        } else {
            collectionUri = MediaStore.Video.Media.EXTERNAL_CONTENT_URI;
        }

        ContentValues values = new ContentValues();
        values.put(MediaStore.Video.Media.DISPLAY_NAME, getSplitFileName(filePath));
        values.put(MediaStore.Video.Media.MIME_TYPE, getMIMEType(filePath));
        values.put(MediaStore.Video.Media.DATE_ADDED, System.currentTimeMillis());
        values.put(MediaStore.Video.Media.DATE_TAKEN, System.currentTimeMillis());

        Uri itemUri = resolver.insert(collectionUri, values);
        try (
                InputStream inputStream = new FileInputStream(filePath);
                OutputStream outputStream = resolver.openOutputStream(itemUri)
        ) {
            byte[] buf = new byte[8192];
            int n;
            while ((n = inputStream.read(buf)) > 0) {
                outputStream.write(buf, 0, n);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static void insertGif(Context context, String filePath) {
        ContentResolver resolver = context.getContentResolver();
        Uri collectionUri;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            collectionUri = MediaStore.Images.Media.getContentUri(MediaStore.VOLUME_EXTERNAL_PRIMARY);
        } else {
            collectionUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI;
        }

        ContentValues values = new ContentValues();
        values.put(MediaStore.Video.Media.DISPLAY_NAME, getSplitFileName(filePath));
        values.put(MediaStore.Video.Media.MIME_TYPE, "image/gif");
        values.put(MediaStore.Video.Media.DATE_ADDED, System.currentTimeMillis());
        values.put(MediaStore.Video.Media.DATE_TAKEN, System.currentTimeMillis());

        Uri itemUri = resolver.insert(collectionUri, values);
        try (
                InputStream inputStream = new FileInputStream(filePath);
                OutputStream outputStream = resolver.openOutputStream(itemUri)
        ) {
            byte[] buf = new byte[8192];
            int n;
            while ((n = inputStream.read(buf)) > 0) {
                outputStream.write(buf, 0, n);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static void insertImage(Context context, String filePath, String fileName) throws FileNotFoundException {
        File file = new File(filePath);
        MediaStore.Images.Media.insertImage(context.getContentResolver(), filePath, fileName, null);
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) {
            Intent intent = new Intent(Intent.ACTION_MEDIA_MOUNTED, Uri.fromFile(file));
            intent.addFlags(Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
            context.sendBroadcast(intent);
        } else {
            Intent intent = new Intent(Intent.ACTION_MEDIA_SCANNER_SCAN_FILE, Uri.fromFile(file));
            intent.addFlags(Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
            context.sendBroadcast(intent);
        }
    }

    private static String getMIMEType(String filePath) {
        try {
            String fileExtension = "mp4";
            int i = filePath.lastIndexOf('.');
            if (i > 0) {
                fileExtension = filePath.substring(i + 1);
            }
            return MimeTypeMap.getSingleton().getMimeTypeFromExtension(fileExtension);
        } catch (Exception e) {
            return "video/mp4";
        }
    }

    private static String getSplitFileName(String filePath) {
        if (!TextUtils.isEmpty(filePath)) {
            String[] split = filePath.split("/");
            if (split.length > 0) {
                return "AIHelp-" + split[split.length - 1];
            }
        }
        return "UnknownFile";
    }

    public static String getTargetSavePath(String filePath, String fileName) {
        String dirPath = "";
        File filesDir = AIHelpContext.getInstance().getContext().getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS);
        if (filesDir != null) {
            dirPath = filesDir.getAbsolutePath() + "/AIHelp";
            File dirFile = new File(dirPath);
            if (!dirFile.exists()) {
                if (dirFile.mkdirs()) {
                    return dirFile.getAbsolutePath() + File.separator + fileName;
                }
            }
        }
        return dirPath + File.separator + fileName;
    }

}
