package net.aihelp.utils;

import android.content.Context;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.text.Spannable;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.text.TextPaint;
import android.text.TextUtils;
import android.text.method.LinkMovementMethod;
import android.text.style.ClickableSpan;
import android.text.style.UnderlineSpan;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.viewer.ImageViewer;
import net.aihelp.data.event.PageHoppingEvent;
import net.aihelp.ui.preview.PreviewActivity;
import net.aihelp.ui.preview.data.PreviewInfo;

import java.io.File;
import java.util.LinkedList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import androidx.fragment.app.Fragment;

public class RichTextSlicer {

    public static View getRichTextView(Context context, Fragment fragment, final String msg) {

        LinearLayout layout = new LinearLayout(context);
        layout.setOrientation(LinearLayout.VERTICAL);
        layout.setGravity(Gravity.START);

        try {
            LinkedList<Integer> urlIndexList = new LinkedList<>();
            urlIndexList.add(0);

            // 使用正则匹配返回的信息，里边可能有表单、图片、普通文字等，并将匹配到的位置信息存到数组中
            Matcher formMatcher = Pattern.compile(RegexUtil.REGEX_RICH_TEXT).matcher(msg);
            while (formMatcher.find()) {
                urlIndexList.add(formMatcher.start());
                urlIndexList.add(formMatcher.end());
            }

            if (urlIndexList.getLast() != msg.length()) {
                urlIndexList.add(msg.length());
            }

            for (int i = 0; i < urlIndexList.size() - 1; i++) {
                final String urlOrRawText = msg.substring(urlIndexList.get(i), urlIndexList.get(i + 1)).trim();
                // 因为 addView 的时候自然会换行，所以不需要处理文本中自带的换行符
                if (Pattern.compile("\\s*?").matcher(urlOrRawText).matches() || ";".equals(urlOrRawText)) {
                    continue;
                }
                // 根据正则筛出图片、视频以及普通文本消息，实现图文混排效果
                if (Pattern.compile(RegexUtil.REGEX_IMAGE).matcher(urlOrRawText).matches()) {
                    layout.addView(getImageViewFromRichText(context, fragment, urlOrRawText));
                } else if (Pattern.compile(RegexUtil.REGEX_VIDEO).matcher(urlOrRawText).matches()) {
                    layout.addView(getVideoViewFromRichText(context, fragment, urlOrRawText));
                } else {
                    layout.addView(getTextViewFromRichText(context, urlOrRawText));
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return layout;
    }

    private static View getImageViewFromRichText(Context context, Fragment fragment, String imageUrl) {
        final ImageView imageView = new ImageView(context);
        LinearLayout.MarginLayoutParams layoutParams = new ViewGroup.MarginLayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT);
        layoutParams.topMargin = Styles.dpToPx(context, 3);
        imageView.setLayoutParams(layoutParams);
        imageView.setAdjustViewBounds(true);
        MediaUtils.scaleImageView(imageUrl, imageView, imageView, null);
        imageView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (fragment != null) {
                    PreviewActivity.startAct(fragment, PreviewInfo.get(imageUrl));
                }
            }
        });
        return imageView;
    }

    private static View getVideoViewFromRichText(Context context, Fragment fragment, String videoUrl) {
        View rootView = View.inflate(context, ResResolver.getLayoutId("aihelp_loading_image_view"), null);
        ImageView imageView = rootView.findViewById(ResResolver.getViewId("aihelp_image_view"));
        ImageView ivPlay = rootView.findViewById(ResResolver.getViewId("aihelp_iv_play"));
        View maskView = rootView.findViewById(ResResolver.getViewId("aihelp_v_mask"));
        View loadingView = rootView.findViewById(ResResolver.getViewId("aihelp_loading_view"));
        rootView.setLayoutParams(new LinearLayout.LayoutParams(Styles.dpToPx(context, 120), Styles.dpToPx(context, 150)));

        MediaUtils.getImageForVideo(videoUrl, new MediaUtils.OnLoadVideoImageListener() {
            @Override
            public void onLoadImage(File file) {
                MediaUtils.scaleImageView(file.getAbsolutePath(), imageView, rootView, new MediaUtils.OnImageScaledListener() {
                    @Override
                    public void onImageScaled() {
                        maskView.setVisibility(View.GONE);
                        loadingView.setVisibility(View.GONE);
                        ivPlay.setVisibility(View.VISIBLE);
                    }
                });

                rootView.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        if (fragment != null) {
                            PreviewActivity.startAct(fragment, PreviewInfo.get(videoUrl));
                        }
                    }
                });
            }
        });

        return rootView;
    }

    private static View getTextViewFromRichText(Context context, final String text) {
        TextView textView = getMsg(context, text);
        textView.setLayoutParams(new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT));
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            textView.setTextDirection(View.TEXT_DIRECTION_LOCALE);
        }
        return textView;
    }

    private static TextView getMsg(Context context, String message) {
        TextView textView = new TextView(context);
        textView.setTextSize(15);
        textView.setLineSpacing(Styles.dpToPx(context, 3), 1.0f);
        textView.setTextColor(Color.parseColor(CustomConfig.CommonSetting.textColor));
        textView.setTextIsSelectable(true);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            textView.setTextDirection(View.TEXT_DIRECTION_LOCALE);
        }

        boolean isMIUI12 = ("XiaoMi".equalsIgnoreCase(Build.MANUFACTURER) || "RedMi".equalsIgnoreCase(Build.MANUFACTURER)) &&
                Build.VERSION.SDK_INT <= Build.VERSION_CODES.R;
        boolean isSamsungOreo = "samsung".equalsIgnoreCase(Build.MANUFACTURER) && Build.VERSION.SDK_INT <= Build.VERSION_CODES.O;

        if (isMIUI12 || isSamsungOreo) {
            textView.setText(message);
        } else {
            textView.setText(getUrlSupportedText(context, message));
            textView.setMovementMethod(LinkMovementMethod.getInstance());
        }

        return textView;
    }

    private static SpannableStringBuilder getUrlSupportedText(Context context, String msg) {
        if (TextUtils.isEmpty(msg)) return new SpannableStringBuilder("");

        SpannableStringBuilder ssb = new SpannableStringBuilder(msg);

        // 匹配多个 https 连接
        Matcher matcher = Pattern.compile("\\S+(://)\\S+").matcher(ssb);
        while (matcher.find()) {
            int start = matcher.start();
            int end = matcher.end();
            final String url = matcher.group();

            ClickableSpan clickableSpan = new ClickableSpan() {
                @Override
                public void onClick(View view) {
                    if (AppInfoUtil.isUrlStillNeedResponding(context, url)) {
                        if (isAttachment(url)) {
                            PreviewActivity.startAct(context, PreviewInfo.get(url));
                        } else {
                            Bundle bundle = new Bundle();
                            bundle.putString(IntentValues.INTENT_URL, url);
                            EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_FORM, bundle));
                        }
                    }
                }

                @Override
                public void updateDrawState(TextPaint dp) {
                    dp.setColor(Color.parseColor(CustomConfig.CommonSetting.interactElementTextColor));
                }
            };
            ssb.setSpan(clickableSpan, start, end, Spannable.SPAN_EXCLUSIVE_INCLUSIVE);
            ssb.setSpan(new UnderlineSpan(), start, end, Spanned.SPAN_EXCLUSIVE_INCLUSIVE);
        }
        return ssb;
    }

    private static boolean isAttachment(String url) {
        String fileTypes = CustomConfig.UploadLimit.fileTypes;
        if (!TextUtils.isEmpty(fileTypes) && !TextUtils.isEmpty(url)) {
            String[] fileExtensions = fileTypes.split(",");
            for (String extension : fileExtensions) {
                if (url.endsWith(extension)) {
                    return true;
                }
            }
        }
        return false;
    }

}
