/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.lang;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.StringTokenizer;

/**
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 */
public class Comparators {

    /**
     * 参与对比的属性
     * @param getter
     * @return
     */
    public static <T> ComparatorHelper<T> orderBy(Getter<T> getter) {
        return new ComparatorHelper<T>().orderBy(getter);
    }

    /**
     * 参与对比的属性
     * @param getter
     * @param asc 为true使用升序，为false使用降序
     * @return
     */
    public static <T> ComparatorHelper<T> orderBy(Getter<T> getter, boolean asc) {
        return new ComparatorHelper<T>().orderBy(getter, asc);
    }

    /**
     * 参与对比的属性
     * @param getter
     * @return
     */
    public static <T> ComparatorHelper<T> ascBy(Getter<T> getter) {
        return orderBy(getter, true);
    }

    /**
     * 参与对比的属性
     * @param getter
     * @return
     */
    public static <T> ComparatorHelper<T> descBy(Getter<T> getter) {
        return orderBy(getter, false);
    }

    /**
     * 对列表进行升序排序，为null的排在前面
     * @param list
     * @param getter
     */
    public static <T> void sort(List<T> list, Getter<T> getter) {
        orderBy(getter).sort(list);
    }

    /**
     * 对列表进行排序，为null的排在前面
     * @param list
     * @param getter
     * @param asc
     */
    public static <T> void sort(List<T> list, Getter<T> getter, boolean asc) {
        orderBy(getter, asc).sort(list);
    }

    /**
     * 对列表进行排序，为null排在后面
     * @param list
     * @param getter
     */
    public static <T> void sortNullLast(List<T> list, Getter<T> getter) {
        orderBy(getter).nullLast().sort(list);
    }

    /**
     * 对列表进行排序，为null排在后面
     * @param list
     * @param getter
     * @param asc
     */
    public static <T> void sortNullLast(List<T> list, Getter<T> getter, boolean asc) {
        orderBy(getter, asc).nullLast().sort(list);
    }

    /**
     * 对列表进行升序排序，为null排在前面
     * @param list
     * @param getter
     */
    public static <T> void ascSort(List<T> list, Getter<T> getter) {
        sort(list, getter, true);
    }

    /**
     * 对列表进行降序排序，为null排在前面
     * @param list
     * @param getter
     */
    public static <T> void descSort(List<T> list, Getter<T> getter) {
        sort(list, getter, false);
    }

    /**
     * 对列表进行升序排序，为null排在后面
     * @param list
     * @param getter
     */
    public static <T> void ascSortNullLast(List<T> list, Getter<T> getter) {
        sortNullLast(list, getter, true);
    }

    /**
     * 对列表进行降序排序，为null排在后面
     * @param list
     * @param getter
     */
    public static <T> void descSortNullLast(List<T> list, Getter<T> getter) {
        sortNullLast(list, getter, false);
    }

    /**
     *
     * @param o1
     * @param o2
     * @return
     */
    public static <T extends Comparable<T>> int compare(T o1, T o2) {
        return doCompare(o1, o2, false);
    }

    /**
     *
     * @param o1
     * @param o2
     * @return
     */
    public static <T extends Comparable<T>> int nullLastCompare(T o1, T o2) {
        return doCompare(o1, o2, true);
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    private static int doCompare(Comparable o1, Comparable o2, boolean nullLast) {
        int v = compareNull(o1, o2, nullLast);
        if (v == 0 && o1 != null && o2 != null) {
            v = o1.compareTo(o2);
        }
        if (v < 0) {
            return -1;
        } else if (v > 0) {
            return 1;
        } else {
            return 0;
        }
    }

    private static <T> int compareNull(T o1, T o2, boolean nullLast) {
        if (o1 == null) {
            if (o2 != null) {
                return nullLast ? 1 : -1;
            }
        } else if (o2 == null) {
            return nullLast ? -1 : 1;
        }
        return 0;
    }

    /**
     * 比较version1和version2，如果version1小于、等于、大于version2分别返回 -1、0、1
     * @param version1
     * @param version2
     * @return 如果version1小于、等于、大于version2分别返回 -1、0、1
     */
    public static int versionCompare(String version1, String version2) {
        if (version1 == null) {
            if (version2 == null) {
                return 0;
            } else {
                return -1;
            }
        } else if (version2 == null) {
            return 1;
        }
        StringTokenizer t1 = new StringTokenizer(version1, "._");
        StringTokenizer t2 = new StringTokenizer(version2, "._");
        while (t1.hasMoreTokens()) {
            if (!t2.hasMoreTokens()) {
                return 1;
            }
            int n1 = Integer.parseInt(t1.nextToken());
            int n2 = Integer.parseInt(t2.nextToken());
            int d = n1 - n2;
            if (d != 0) {
                return d;
            }
        }
        return t2.hasMoreTokens() ? -1 : 0;
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     * @param <T>
     */
    public static class ComparatorHelper<T> implements Comparator<T> {

        private final List<GetterEntry<T>> entries;

        private boolean nullLast = false;

        private ComparatorHelper() {
            entries = new ArrayList<>();
        }

        /**
         * 参与对比的属性
         * @param getter
         * @return
         */
        public ComparatorHelper<T> orderBy(Getter<T> getter) {
            entries.add(new GetterEntry<>(getter));
            return this;
        }

        /**
         * 参与对比的属性
         * @param getter
         * @param asc 为true使用升序，为false使用降序
         * @return
         */
        public ComparatorHelper<T> orderBy(Getter<T> getter, boolean asc) {
            entries.add(new GetterEntry<>(getter, asc));
            return this;
        }

        /**
         * 参与对比的属性
         * @param getter
         * @return
         */
        public ComparatorHelper<T> ascBy(Getter<T> getter) {
            return orderBy(getter, true);
        }

        /**
         * 参与对比的属性
         * @param getter
         * @return
         */
        public ComparatorHelper<T> descBy(Getter<T> getter) {
            return orderBy(getter, false);
        }

        /**
         * 为null排在后面
         * @return
         */
        public ComparatorHelper<T> nullLast() {
            nullLast = true;
            return this;
        }

        /**
         * 对指定的列表进行排序
         * @param list
         */
        public void sort(List<T> list) {
            Collections.sort(list, this);
        }

        @Override
        public int compare(T o1, T o2) {
            int v = Comparators.compareNull(o1, o2, nullLast);
            if (v == 0) {
                for (GetterEntry<T> entry : entries) {
                    v = compare(entry, o1, o2);
                    if (v != 0) {
                        break;
                    }
                }
            }
            return v;
        }

        private int compare(GetterEntry<T> entry, T o1, T o2) {
            Comparable<?> c1 = entry.getter.apply(o1);
            Comparable<?> c2 = entry.getter.apply(o2);
            int v = Comparators.doCompare(c1, c2, nullLast);
            if (entry.asc) {
                return v;
            } else {
                return -v;
            }
        }

    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     * @param <T>
     */
    private static class GetterEntry<T> {

        final Getter<T> getter;

        final boolean asc;

        GetterEntry(Getter<T> getter) {
            this(getter, true);
        }

        GetterEntry(Getter<T> getter, boolean asc) {
            this.getter = getter;
            this.asc = asc;
        }

    }

    /**
     *
     * @param <T>
     */
    public interface Getter<T> {

        Comparable<?> apply(T o);

    }
}
