/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.lang;

import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * 
 * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
 *
 */
public final class Dates {
    
    private Dates() {}
    
    /**
     * 获取网络时间戳
     * @return 返回网络时间戳
     * @throws IOException 访问网络失败时抛出
     */
    public static long networkTimeMillis() throws IOException {
        return networkTimeMillis("http://www.baidu.com", "http://www.google.com", "http://www.taobao.com");
    }
    
    /**
     * 从指定的网络地址获取网络时间戳
     * @param urls 网址
     * @return 返回网络时间戳
     * @throws IOException 访问网络失败时抛出
     */
    public static long networkTimeMillis(String... urls) throws IOException {
        return networkTimeMillis(3, urls);
    }
    
    /**
     * 从指定的网络地址获取网络时间戳
     * @param connectTimeout 连接超时时间
     * @param urls 网址
     * @return 返回网络时间戳
     * @throws IOException 访问网络失败时抛出
     */
    public static long networkTimeMillis(int connectTimeout, String... urls) throws IOException {
        IOException exception = null;
        for (String webUrl : urls) {
            try {
                URL url = new URL(webUrl);
                URLConnection uc = url.openConnection();
                if (connectTimeout > 0) {
                    uc.setConnectTimeout(connectTimeout);
                }
                uc.connect();
                long timeMs = uc.getDate();
                if (timeMs > 0) {
                    return timeMs;
                }
            } catch (IOException e) {
                exception = e;
            }
        }
        if (exception == null) {
            exception = new IOException("Connect failed.");
        }
        throw exception;
    }
    
    /**
     * 计算两日期相差的天数（自然天数），即 to - from
     * 
     * @param from 开始日期
     * @param to 结束日期
     * @return 返回相差的天数，即结束日期减开始日期得到的自然天数
     */
    public static int dayDiff(Date from, Date to) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        try {
            from = df.parse(df.format(from));
            to = df.parse(df.format(to));
        } catch (Exception ex) {
            throw new RuntimeException();
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(to);
        long toTime = calendar.getTimeInMillis() + calendar.get(Calendar.DST_OFFSET);
        toTime /= 1000;
        calendar.setTime(from);
        long fromTime = calendar.getTimeInMillis() + calendar.get(Calendar.DST_OFFSET);
        fromTime /= 1000;
        long days = Math.round((toTime - fromTime) / (24 * 60 * 60.0));
        return (int)days;
    }

    /**
     * 判断指定的两个日期是否是同一天
     *
     * @param date1
     * @param date2
     * @return 如果是同一天返回true，否则返回false
     */
    public static boolean isSameDay(Date date1, Date date2) {
        if (date1 != null && date2 != null) {
            Calendar c1 = Calendar.getInstance();
            c1.setTime(date1);
            Calendar c2 = Calendar.getInstance();
            c2.setTime(date2);
            
            return (c1.get(Calendar.YEAR) == c2.get(Calendar.YEAR))
                    && (c1.get(Calendar.MONTH) == c2.get(Calendar.MONTH))
                    && (c1.get(Calendar.DAY_OF_MONTH) == c2.get(Calendar.DAY_OF_MONTH));
        }
        return false;
    }
    
    /**
     * 判断指定的日期段与指定的月日段是否重叠
     * @param startDate1
     * @param endDate1
     * @param startDate2
     * @param endDate2
     * @param ignoreYearS1 忽略startDate1中的年份
     * @param ignoreYearE1 忽略endDate1中的年份
     * @param ignoreYearS2 忽略startDate2中的年份
     * @param ignoreYearE2 忽略endDate2中的年份
     * @return 重叠返回true，否则返回false
     */
    public static boolean isDateOverlap(Date startDate1, Date endDate1, Date startDate2, Date endDate2,
            boolean ignoreYearS1, boolean ignoreYearE1, boolean ignoreYearS2, boolean ignoreYearE2) {
        int iStartDate1;
        int iStartDate2;
        SimpleDateFormat ymdFmt = new SimpleDateFormat("yyyyMMdd");
        SimpleDateFormat mdFmt = new SimpleDateFormat("MMdd");
        if (ignoreYearS1 != ignoreYearS2) {
            int year;
            if (ignoreYearS1) {
                year = toYear(startDate2);
            } else {
                year = toYear(startDate1);
            }
            iStartDate1 = Integer.valueOf(year + mdFmt.format(startDate1));
            iStartDate2 = Integer.valueOf(year + mdFmt.format(startDate2));
        } else if (ignoreYearS1) { // ignoreYearS1 && ignoreYearS2
            iStartDate1 = Integer.valueOf("1" + mdFmt.format(startDate1));
            iStartDate2 = Integer.valueOf("1" + mdFmt.format(startDate2));
        } else { // !ignoreYearS1 && !ignoreYearS2
            iStartDate1 = Integer.valueOf(ymdFmt.format(startDate1));
            iStartDate2 = Integer.valueOf(ymdFmt.format(startDate2));
        }
        int iEndDate1;
        int iEndDate2;
        if (ignoreYearE1 != ignoreYearE2) {
            int year;
            if (ignoreYearE1) {
                year = toYear(endDate2);
            } else {
                year = toYear(endDate1);
            }
            iEndDate1 = Integer.valueOf(year + mdFmt.format(endDate1));;
            iEndDate2 = Integer.valueOf(year + mdFmt.format(endDate2));
        } else if (ignoreYearE1) { // ignoreYearE1 && ignoreYearE2
            iEndDate1 = Integer.valueOf("1" + mdFmt.format(endDate1));
            iEndDate2 = Integer.valueOf("1" + mdFmt.format(endDate2));
        } else { // !ignoreYearE1 && !ignoreYearE2
            iEndDate1 = Integer.valueOf(ymdFmt.format(endDate1));
            iEndDate2 = Integer.valueOf(ymdFmt.format(endDate2));
        }
        return iStartDate1 <= iStartDate2 && iStartDate2 <= iEndDate1
                || iStartDate1 <= iEndDate2 && iEndDate2 <= iEndDate1
                || iStartDate2 < iStartDate1 && iEndDate1 < iEndDate2;
    }
    
    private static int toYear(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return calendar.get(Calendar.YEAR);
    }

}
