/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.lang;

import javax.net.ssl.*;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.NetworkInterface;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Enumeration;

/**
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 */
public final class Networks {
    private Networks() {
    }

    /**
     * 获取本机的MAC地址的字符串格式。xx:xx:xx:xx
     * @return
     */
    public static String macString() {
        byte[] macBytes = macBytes();
        if (macBytes == null) {
            return null;
        }
        StringBuilder buf = new StringBuilder();
        for (byte b : macBytes) {
            buf.append(Bytes.toHex(b));
            buf.append(":");
        }
        buf.setLength(buf.length() - 1);
        return buf.toString();
    }

    /**
     * 获取本机的MAC地址的Long值
     * @return
     */
    public static Long macValue() {
        byte[] macBytes = macBytes();
        if (macBytes == null) {
            return null;
        }
        byte[] bytes = new byte[8];
        System.arraycopy(macBytes, 0, bytes, 2, 6);
        return Bytes.bytesToLong(bytes);
    }

    public static byte[] macBytes() {
        try {
            Enumeration<NetworkInterface> en = NetworkInterface.getNetworkInterfaces();
            while (en.hasMoreElements()) {
                NetworkInterface nint = en.nextElement();
                if (!nint.isLoopback()) {
                    byte[] data = nint.getHardwareAddress();
                    if (data != null && data.length == 6) {
                        return data;
                    }
                }
            }
        } catch (java.net.SocketException e) {
            // fine, let's take is as signal of not having any interfaces
        }
        return null;
    }

    public static String url(String host, String path) {
        if (host.endsWith("/")) {
            if (path.startsWith("/")) {
                return host + path.substring(1);
            }
        } else if (!path.startsWith("/")) {
            return host + "/" + path;
        }
        return host + path;
    }

    public static HttpURLConnection connectExemptSSL(URL url)
            throws IOException, KeyManagementException, NoSuchAlgorithmException {
        return connect(url, null, null);
    }

    public static HttpURLConnection connect(URL url, SSLContext sslContext, HostnameVerifier hostNameVerifier)
            throws IOException, KeyManagementException, NoSuchAlgorithmException {
        HttpURLConnection conn = (HttpURLConnection) url.openConnection();
        if (HttpsURLConnection.class.isInstance(conn)) {
            HttpsURLConnection https = HttpsURLConnection.class.cast(conn);
            if (hostNameVerifier == null) {
                hostNameVerifier = new HostnameVerifier() {
                    @Override
                    public boolean verify(String s, SSLSession sslSession) {
                        return true;
                    }
                };
            }
            https.setHostnameVerifier(hostNameVerifier);
            if (sslContext == null) {
                sslContext = exemptSSLContext();
            }
            https.setSSLSocketFactory(sslContext.getSocketFactory());
        }
        return conn;
    }

    /**
     * 创建一个支持双向认证的SSLContext
     *
     * @param trustCert
     * @param keyCert
     * @param keyCertPassword
     * @return
     * @throws Exception
     */
    public static SSLContext mutualAuthSSLContext(String trustCert, String keyCert, String keyCertPassword)
            throws Exception {
        KeyManager[] keyManagers = certKeyManagers(keyCert, keyCertPassword);
        TrustManager[] trustManagers = certTrustManagers(trustCert);
        SSLContext sslContext = SSLContext.getInstance("TLS");
        sslContext.init(keyManagers, trustManagers, new SecureRandom());
        return sslContext;
    }

    /**
     * 创建一个不检查证书的SSLContext
     *
     * @return 返回一个不检查证书的SSLContext
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     */
    public static SSLContext exemptSSLContext() throws NoSuchAlgorithmException, KeyManagementException {
        SSLContext context = SSLContext.getInstance("TLS");
        context.init(null, ExemptVerifyTrustManager.trustManagers, new SecureRandom());
        return context;
    }

    /**
     * 使用base64格式的证书创建KeyManager数组
     *
     * @param pemCert
     * @param certPassword
     * @return
     * @throws Exception
     */
    public static KeyManager[] certKeyManagers(String pemCert, String certPassword) throws Exception {
        byte[] bytes = Base64.decode(pemCert);
        KeyStore keySotre = KeyStore.getInstance("PKCS12");
        keySotre.load(new ByteArrayInputStream(bytes), certPassword.toCharArray());
        KeyManagerFactory kmf = KeyManagerFactory.getInstance("SunX509");
        kmf.init(keySotre, certPassword.toCharArray());
        return kmf.getKeyManagers();
    }

    /**
     * 使用base64格式的证书创建TrustManager数组
     *
     * @param pemCert
     * @return
     * @throws Exception
     */
    public static TrustManager[] certTrustManagers(String pemCert) throws Exception {
        byte[] bytes = Base64.decode(pemCert);
        CertificateFactory factory = CertificateFactory.getInstance("X.509");
        X509Certificate cert = (X509Certificate) factory.generateCertificate(new ByteArrayInputStream(bytes));
        String alias = cert.getSubjectX500Principal().getName();
        KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
        trustStore.load(null);
        trustStore.setCertificateEntry(alias, cert);
        TrustManagerFactory tmf = TrustManagerFactory.getInstance("X509");
        tmf.init(trustStore);
        return tmf.getTrustManagers();
    }

    /**
     * 创建一个不检查证书的TrustManager[]
     * @return
     */
    public static TrustManager[] exemptTrustManagers() {
        return ExemptVerifyTrustManager.trustManagers;
    }

    /**
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    private static class ExemptVerifyTrustManager implements X509TrustManager {
        static TrustManager[] trustManagers = new TrustManager[]{new ExemptVerifyTrustManager()};
        private static final X509Certificate[] _X509S = new X509Certificate[]{};

        @Override
        public void checkClientTrusted(X509Certificate[] x509s, String s) {
        }

        @Override
        public void checkServerTrusted(X509Certificate[] x509s, String s) {
        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return _X509S;
        }
    }

}
