/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.net;

import net.apexes.commons.lang.Checks;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;

/**
 * 
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 *
 * @param <R> Response的数据类型
 */
public class JsonHttpCaller<R> {

    protected final StringHttpCaller caller;
    private final JsonEncoder jsonEncoder;
    private final JsonDecoder<R> jsonDecoder;
    private RequestEncoder requestEncoder;

    public JsonHttpCaller(String url, JsonEncoder encoder, JsonDecoder<R> decoder) {
        Checks.verifyNotNull(encoder, "encoder");
        Checks.verifyNotNull(decoder, "decoder");
        this.caller = new StringHttpCaller(url);
        this.jsonEncoder = encoder;
        this.jsonDecoder = decoder;
    }

    private JsonHttpCaller(String url, JsonEncoder encoder) {
        Checks.verifyNotNull(encoder, "encoder");
        this.caller = new StringHttpCaller(url);
        this.jsonEncoder = encoder;
        this.jsonDecoder = null;
    }

    public JsonHttpCaller<R> setAcceptCompress(boolean value) {
        caller.setAcceptCompress(value);
        return this;
    }

    public JsonHttpCaller<R> setConnectTimeout(int timeout) {
        caller.setConnectTimeout(timeout);
        return this;
    }

    public JsonHttpCaller<R> setReadTimeout(int timeout) {
        caller.setReadTimeout(timeout);
        return this;
    }

    public JsonHttpCaller<R> setSslContext(SSLContext sslContext) {
        caller.setSslContext(sslContext);
        return this;
    }

    public JsonHttpCaller<R> setHostNameVerifier(HostnameVerifier hostNameVerifier) {
        caller.setHostNameVerifier(hostNameVerifier);
        return this;
    }

    public JsonHttpCaller<R> setHttpProperty(String key, String value) {
        caller.setHttpProperty(key, value);
        return this;
    }

    public JsonHttpCaller<R> setRequestLogger(StringHttpCaller.CallLogger requestLogger) {
        caller.setRequestLogger(requestLogger);
        return this;
    }

    public JsonHttpCaller<R> setResponseLogger(StringHttpCaller.CallLogger responseLogger) {
        caller.setResponseLogger(responseLogger);
        return this;
    }

    public JsonHttpCaller<R> setRequestEncoder(RequestEncoder requestEncoder) {
        caller.setRequestEncoder(new RequestEncoderAdapter(this, requestEncoder));
        return this;
    }

    public JsonHttpCaller<R> setResponseReader(StringHttpCaller.ResponseReader responseReader) {
        caller.setResponseReader(responseReader);
        return this;
    }

    public R call(Object request) throws Exception {
        return call(request, false);
    }

    public R call(Object request, boolean compress) throws Exception {
        String requestJson = encode(request);
        String responseJson = caller.doPost(requestJson, compress);
        return decode(responseJson);
    }

    public R callGet() throws Exception {
        String responseJson = caller.doGet();
        return decode(responseJson);
    }

    protected String encode(Object request) throws Exception {
        String requestJson = null;
        if (request != null) {
            if (request instanceof String) {
                requestJson = (String) request;
            } else {
                requestJson = jsonEncoder.toJson(request);
            }
        }
        return requestJson;
    }

    protected R decode(String responseJson) throws Exception {
        return jsonDecoder.fromJson(responseJson);
    }

    public static JsonHttpNoticer forNotice(String url, JsonEncoder jsonEncoder) {
        return new JsonHttpNoticer(url, jsonEncoder);
    }

    public static <R> JsonHttpCaller<R> forRequest(String url, JsonEncoder jsonEncoder, JsonDecoder<R> jsonDecoder) {
        return new JsonHttpCaller<>(url, jsonEncoder, jsonDecoder);
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    public static class JsonHttpNoticer extends JsonHttpCaller<Void> {

        protected JsonHttpNoticer(String url, JsonEncoder jsonEncoder) {
            super(url, jsonEncoder);
        }

        public void notice(Object request) throws Exception {
            notice(request, false);
        }

        public void notice(Object request, boolean compress) throws Exception {
            String requestJson = encode(request);
            caller.doPost(requestJson, compress);
        }

    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    private static class RequestEncoderAdapter implements StringHttpCaller.RequestEncoder {

        private final JsonHttpCaller jsonHttpCaller;
        private final RequestEncoder encoder;

        private RequestEncoderAdapter(JsonHttpCaller jsonHttpCaller, RequestEncoder encoder) {
            this.jsonHttpCaller = jsonHttpCaller;
            this.encoder = encoder;
        }

        @Override
        public byte[] encode(StringHttpCaller caller, String request, boolean compress) throws Exception {
            return encoder.encode(jsonHttpCaller, request, compress);
        }
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    public interface JsonEncoder {
        String toJson(Object object) throws Exception;
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     * @param <R>
     */
    public interface JsonDecoder<R> {
        R fromJson(String json) throws Exception;
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    public interface RequestEncoder {
        byte[] encode(JsonHttpCaller caller, String request, boolean compress) throws Exception;
    }
}
