/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.net;

import net.apexes.commons.lang.Checks;
import net.apexes.commons.lang.Networks;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.SocketException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * 
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 *
 */
public class StringHttpCaller {

    public static final String CONTENT_TYPE_APPLICATION_JSON = "application/json; charset=utf-8";
    public static final String CONTENT_TYPE_APPLICATION_OCTETSTREAM = "application/octet-stream";
    
    public static final String METHOD_POST = "POST";
    public static final String METHOD_GET = "GET";
    public static final String PROPERTY_CONTENT_ENCODING = "Content-Encoding";
    public static final String PROPERTY_ACCEPT_ENCODING = "Accept-Encoding";
    public static final String PROPERTY_CONNECTION = "Connection";
    public static final String PROPERTY_CONTENT_TYPE = "Content-Type";
    public static final String CONNECTION_CLOSE = "close";
    public static final String GZIP = "gzip";

    private static final RequestEncoder DEFAULT_REQUEST_ENCODER = new SimpleRequestEncoder();

    private final String url;
    private final String contentType;
    private final Map<String, String> httpProperties;
    private int connectTimeout = 3 * 1000;
    private int readTimeout = 180 * 1000;
    private boolean acceptCompress;
    private SSLContext sslContext;
    private HostnameVerifier hostNameVerifier;
    private CallLogger requestLogger;
    private CallLogger responseLogger;
    private RequestEncoder requestEncoder;
    private ResponseReader responseReader;

    public StringHttpCaller(String url) {
        this(url, CONTENT_TYPE_APPLICATION_JSON);
    }

    public StringHttpCaller(String url, String contentType) {
        Checks.verifyNotNull(url, "url");
        Checks.verifyNotNull(contentType, "contentType");
        this.url = url;
        this.contentType = contentType;
        httpProperties = new LinkedHashMap<>();
    }

    public boolean isAcceptCompress() {
        return acceptCompress;
    }

    public StringHttpCaller setAcceptCompress(boolean value) {
        acceptCompress = value;
        return this;
    }
    
    /**
     * 
     * @param timeout 超时时间，单位ms
     * @return
     */
    public StringHttpCaller setConnectTimeout(int timeout) {
        this.connectTimeout = timeout;
        return this;
    }
    
    /**
     * 
     * @param timeout 超时时间，单位ms
     * @return
     */
    public StringHttpCaller setReadTimeout(int timeout) {
        this.readTimeout = timeout;
        return this;
    }
    
    /**
     * 
     * @param sslContext
     * @return
     */
    public StringHttpCaller setSslContext(SSLContext sslContext) {
        this.sslContext = sslContext;
        return this;
    }
    
    /**
     * 
     * @param hostNameVerifier
     * @return
     */
    public StringHttpCaller setHostNameVerifier(HostnameVerifier hostNameVerifier) {
        this.hostNameVerifier = hostNameVerifier;
        return this;
    }

    public StringHttpCaller setHttpProperty(String key, String value) {
        Checks.verifyNotNull(key, "key");
        Checks.verifyNotNull(value, "value");
        httpProperties.put(key, value);
        return this;
    }
    
    public StringHttpCaller setRequestLogger(CallLogger requestLogger) {
        this.requestLogger = requestLogger;
        return this;
    }
    
    public StringHttpCaller setResponseLogger(CallLogger responseLogger) {
        this.responseLogger = responseLogger;
        return this;
    }

    public StringHttpCaller setRequestEncoder(RequestEncoder requestEncoder) {
        this.requestEncoder = requestEncoder;
        return this;
    }
    
    public StringHttpCaller setResponseReader(ResponseReader responseReader) {
        this.responseReader = responseReader;
        return this;
    }

    public String doPost(String request) throws Exception {
        return doPost(request, false);
    }

    public String doPost(String request, boolean compress) throws Exception {
        byte[] body;
        if (requestEncoder != null) {
            body = requestEncoder.encode(this, request, compress);
        } else {
            body = DEFAULT_REQUEST_ENCODER.encode(this, request, compress);
        }
        HttpURLConnection conn = openConnection(METHOD_POST, new URL(url), compress);
        try {
            if (requestLogger != null) {
                requestLogger.log(request);
            }
            if (body != null) {
                conn.getOutputStream().write(body);
                conn.getOutputStream().close();
            }
            return handleResponse(conn);
        } finally {
            conn.disconnect();
        }
    }

    public String doGet() throws Exception {
        HttpURLConnection conn = openConnection(METHOD_GET, new URL(url), false);
        try {
            if (requestLogger != null) {
                requestLogger.log(url);
            }
            conn.connect();
            return handleResponse(conn);
        } finally {
            conn.disconnect();
        }
    }
    
    private String handleResponse(HttpURLConnection conn) throws Exception {
        if (conn.getResponseCode() != 200) {
            throw new ResponseException(conn.getResponseCode(), conn.getResponseMessage());
        }
        InputStream ips;
        if (GZIP.equalsIgnoreCase(conn.getHeaderField(PROPERTY_CONTENT_ENCODING))) {
            ips = new GZIPInputStream(conn.getInputStream());
        } else {
            ips = conn.getInputStream();
        }
        if (responseReader == null) {
            responseReader = new SimpleResponseReader();
        }
        String responseJson = responseReader.read(ips);
        if (requestLogger != null) {
            responseLogger.log(responseJson);
        }
        return responseJson;
    }
    
    protected HttpURLConnection openConnection(String method, URL url, boolean compress) throws Exception {
        HttpURLConnection conn = (HttpURLConnection) url.openConnection();
        conn.setRequestMethod(method);
        conn.setDoOutput(true);
        conn.setDoInput(true);
        conn.setRequestProperty(PROPERTY_CONNECTION, CONNECTION_CLOSE);
        
        if (HttpsURLConnection.class.isInstance(conn)) {
            HttpsURLConnection https = HttpsURLConnection.class.cast(conn);
            if (hostNameVerifier != null) {
                https.setHostnameVerifier(hostNameVerifier);
            }
            if (sslContext == null) {
                sslContext = Networks.exemptSSLContext();
            }
            https.setSSLSocketFactory(sslContext.getSocketFactory());
        }
        
        conn.setConnectTimeout(connectTimeout);
        conn.setReadTimeout(readTimeout);
        if (compress) {
            conn.setRequestProperty(PROPERTY_CONTENT_ENCODING, GZIP);
            conn.setRequestProperty(PROPERTY_CONTENT_TYPE, CONTENT_TYPE_APPLICATION_OCTETSTREAM);
        } else {
            conn.setRequestProperty(PROPERTY_CONTENT_TYPE, contentType);
        }
        if (acceptCompress) {
            conn.setRequestProperty(PROPERTY_ACCEPT_ENCODING, GZIP);
        }
        
        for (Map.Entry<String, String> entry : httpProperties.entrySet()) {
            conn.setRequestProperty(entry.getKey(), entry.getValue());
        }
        
        return conn;
    }
    
    public static StringHttpCaller forRequest(String url) {
        return new StringHttpCaller(url);
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    public interface RequestEncoder {
        byte[] encode(StringHttpCaller caller, String request, boolean compress) throws Exception;
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    public static class SimpleRequestEncoder implements RequestEncoder {

        @Override
        public byte[] encode(StringHttpCaller caller, String request, boolean compress) throws Exception {
            if (request == null) {
                return null;
            }
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            if (compress) {
                GZIPOutputStream gzipOut = new GZIPOutputStream(out);
                gzipOut.write(request.getBytes("utf-8"));
                gzipOut.finish();
                gzipOut.close();
            } else {
                out.write(request.getBytes("utf-8"));
            }
            return out.toByteArray();
        }
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    public static class SimpleResponseReader implements ResponseReader {

        @Override
        public String read(InputStream ips) throws Exception {
            BufferedReader reader = new BufferedReader(new InputStreamReader(ips, "utf-8"));
            try {
                StringBuilder sb = new StringBuilder();
                String line;
                while ((line = reader.readLine()) != null) {
                    sb.append(line);
                }
                return sb.toString();
            } finally {
                reader.close();
            }
        }
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    public interface ResponseReader {
        String read(InputStream ips) throws Exception;
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     */
    public interface CallLogger {
        void log(String text);
    }

    /**
     *
     * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
     *
     */
    public static class ResponseException extends SocketException {

        private static final long serialVersionUID = 1L;
        private final int responseCode;
        private final String responseMessage;

        public ResponseException(int responseCode, String responseMessage) {
            super(responseCode + " " + responseMessage);
            this.responseCode = responseCode;
            this.responseMessage = responseMessage;
        }

        public int getResponseCode() {
            return responseCode;
        }

        public String getResponseMessage() {
            return responseMessage;
        }
    }

}
