/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.i18n;

import java.text.MessageFormat;
import java.util.Locale;

/**
 * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
 */
public class I18nMsg {

    protected final Logger logger;
    protected final I18nResource resource;

    public I18nMsg(I18nResource resource) {
        this(resource, new IgnoreLogger());
    }

    public I18nMsg(I18nResource resource, Logger logger) {
        this.resource = resource;
        this.logger = logger;
    }

    public Locale getLocale() {
        return resource.getLocale();
    }

    public String getMessage(String key) {
        return getMessage(key, key);
    }

    public String getMessage(String key, String defaultValue) {
        String value = null;
        try {
            value = resource.getString(key);
        } catch (Exception e) {
            logger.logNotFoundKey(key);
        }
        if (value == null) {
            value = defaultValue;
        }
        return value;
    }

    public String formatKey(String key, Object... params) {
        return format(getMessage(key), params);
    }

    /**
     *
     * @param format
     * @param params
     * @return
     */
    private String format(String format, Object... params) {
        if (params != null && params.length > 0) {
            // MessageFormat 要求 It's 中的  ' 是 ''
            try {
                String pattern = format.replaceAll("'", "''");
                return MessageFormat.format(pattern, params);
            } catch (Exception e) {
                logger.logFormatError(e, format, params);
            }
        }
        return format;
    }

    /**
     * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
     */
    public interface Logger {

        void logNotFoundKey(String key);

        void logFormatError(Exception e, String format, Object... params);

    }

    /**
     *
     * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
     */
    private static class IgnoreLogger implements Logger {

        @Override
        public void logNotFoundKey(String key) {
        }

        @Override
        public void logFormatError(Exception e, String format, Object... params) {
        }
    }

}
