/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.lang;

import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.BitSet;
import java.util.Random;
import java.util.concurrent.atomic.AtomicLong;

/**
 *
 * @author <a href=mailto:hedyn@foxmail.com>HeDYn</a>
 */
public class IDGenerator {

    private final byte[] idedBytes;

    private IDGenerator(byte[] idedBytes) {
        this.idedBytes = idedBytes;
    }

    public String gen() {
        return genId(idedBytes);
    }

    static String genId() {
        return genId(SYSTEM_IDED_BYTES);
    }

    private static String genId(byte[] idedBytes) {
        ByteBuffer buf = ByteBuffer.allocate(16);
        buf.putLong(timeMillis());
        buf.put(idedBytes);
        return IDs.toBase58Id(buf.array());
    }

    private static long timeMillis() {
        long timeMillis = System.currentTimeMillis();
        timeMillis <<= 13;
        while (true) {
            long current = lastTime.get();
            if (timeMillis > current) {
                if (lastTime.compareAndSet(current, timeMillis)) {
                    break;
                }
            } else {
                if (lastTime.compareAndSet(current, current + 1)) {
                    timeMillis = current + 1;
                    break;
                }
            }
        }
        return timeMillis;
    }

    static IDGenerator namespace(long namespace) {
        return namespace(Bytes.longToBytes(namespace));
    }

    static IDGenerator namespace(byte[] namespaceBytes) {
        byte[] idedBytes = new byte[8];
        if (namespaceBytes.length > 0) {
            int len = Math.min(namespaceBytes.length, 6);
            int srcPos = namespaceBytes.length - len;
            int destPos = 6 - len;
            System.arraycopy(namespaceBytes, srcPos, idedBytes, destPos, len);
        }
        idedBytes[6] = PID_BYTES[0];
        idedBytes[7] = PID_BYTES[1];
        return new IDGenerator(idedBytes);
    }

    private static final AtomicLong lastTime = new AtomicLong(Long.MIN_VALUE);
    private static final byte[] PID_BYTES;
    private static final byte[] SYSTEM_IDED_BYTES = new byte[8];
    
    static {
        Long macValue = Networks.macValue();
        if (macValue == null) {
            try {
                byte[] localBytes = InetAddress.getLocalHost().getAddress();
                int localValue = Bytes.bytesToInt(localBytes);
                localValue <<= 8;
                macValue = Long.valueOf(localValue);
            } catch (Exception e) {
                Random random = new Random(System.currentTimeMillis());
                macValue = random.nextLong();
            }
        }

        short pid = ProcessIdentifier.Impl.processId();
        PID_BYTES = Bytes.shortToBytes(pid);

        BitSet bitSet = new BitSet(64);
        long m = 1;
        for (int i = 0; i < 16; i++) {
            if ((pid & m) != 0) {
                bitSet.set(i * 4);
            }
            m <<= 1;
        }
        m = 1;
        for (int i =0; i < 48; i++) {
            if ((macValue & m) != 0) {
                bitSet.set(i + (i / 3) + 1);
            }
            m <<= 1;
        }
        byte[] bytes = bitSet.toByteArray();
        System.arraycopy(bytes, 0, SYSTEM_IDED_BYTES, 0, bytes.length);
    }
}
