/*
 * Copyright (c) 2018, apexes.net. All rights reserved.
 *
 *         http://www.apexes.net
 *
 */
package net.apexes.commons.querydsl.info;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * 
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 *
 */
public class TableInfo {

    private static final int FK_FOREIGN_COLUMN_NAME = 8;

    private static final int FK_FOREIGN_TABLE_NAME = 7;

    private static final int FK_FOREIGN_SCHEMA_NAME = 6;

    private static final int FK_NAME = 12;

    private static final int FK_PARENT_COLUMN_NAME = 4;

    private static final int FK_PARENT_TABLE_NAME = 3;

    private static final int FK_PARENT_SCHEMA_NAME = 2;

    private static final int PK_COLUMN_NAME = 4;

    private static final int PK_NAME = 6;

    private final String catalog;
    private final String schema;
    private final String tableName;
    private final boolean lowerCase;
    private final PrimaryKeyInfo primaryKey;
    private final List<ColumnInfo> columns;
    private final List<ForeignKeyInfo> importedKeys;
    private final List<ForeignKeyInfo> exportedKeys;
    private final List<IndexInfo> indexs;

    public TableInfo(DatabaseMetaData md, String catalog, String schema, String tableName)
            throws SQLException {
        this(md, catalog, schema, tableName, false);
    }

    public TableInfo(DatabaseMetaData md, String catalog, String schema, String tableName, boolean lowerCase)
            throws SQLException {
        this.catalog = catalog;
        this.schema = schema;
        this.tableName = tableName;
        this.lowerCase = lowerCase;
        primaryKey = buildPrimaryKey(md);
        columns = buildColumn(md);
        importedKeys = buildImportedKeys(md);
        exportedKeys = buildExportedKeys(md);
        indexs = new ArrayList<>();
        buildUniqueKeysAndIndexs(md);
    }

    private PrimaryKeyInfo buildPrimaryKey(DatabaseMetaData md) throws SQLException {
        ResultSet rs = md.getPrimaryKeys(catalog, schema, tableName);
        try {
            PrimaryKeyInfo pkInfo = null;
            while (rs.next()) {
                String columnName = normalizeSQLName(rs.getString(PK_COLUMN_NAME));
                if (pkInfo == null) {
                    String name = rs.getString(PK_NAME);
                    if (name == null || name.isEmpty()) {
                        name = tableName + "_PK";
                    }
                    pkInfo = new PrimaryKeyInfo(name, schema, tableName);
                }
                pkInfo.add(columnName);
            }
            return pkInfo;
        } finally {
            rs.close();
        }
    }

    private List<ColumnInfo> buildColumn(DatabaseMetaData md) throws SQLException {
        ResultSet rs = md.getColumns(catalog, schema, tableName.replace("/", "//"), null);
        try {
            List<ColumnInfo> list = new ArrayList<>();
            while (rs.next()) {
                String columnName = normalize(rs.getString("COLUMN_NAME"));
                String normalizedColumnName = normalizeSQLName(columnName);
                int columnType = rs.getInt("DATA_TYPE");
                String typeName = rs.getString("TYPE_NAME");
                Integer columnSize = (Integer) rs.getObject("COLUMN_SIZE");
                Integer columnDigits = (Integer) rs.getObject("DECIMAL_DIGITS");
                int columnIndex = rs.getInt("ORDINAL_POSITION");
                int nullable = rs.getInt("NULLABLE");
                String defaultValue = rs.getString("COLUMN_DEF");
                String describe = rs.getString("REMARKS");
                list.add(new ColumnInfo(columnName, normalizedColumnName, columnType, typeName, columnSize,
                        columnDigits, columnIndex, nullable, defaultValue, describe));
            }
            return list;
        } finally {
            rs.close();
        }
    }

    private List<ForeignKeyInfo> buildImportedKeys(DatabaseMetaData md) throws SQLException {
        ResultSet rs = md.getImportedKeys(catalog, schema, tableName);
        try {
            Map<String, ForeignKeyInfo> map = new TreeMap<>();
            while (rs.next()) {
                String name = rs.getString(FK_NAME);
                String parentSchemaName = normalizeSQLName(rs.getString(FK_PARENT_SCHEMA_NAME));
                String parentTableName = normalizeSQLName(rs.getString(FK_PARENT_TABLE_NAME));
                String parentColumnName = normalizeSQLName(rs.getString(FK_PARENT_COLUMN_NAME));
                String foreignSchemaName = normalizeSQLName(rs.getString(FK_FOREIGN_SCHEMA_NAME));
                String foreignTableName = normalizeSQLName(rs.getString(FK_FOREIGN_TABLE_NAME));
                String foreignColumn = normalizeSQLName(rs.getString(FK_FOREIGN_COLUMN_NAME));
                if (name == null || name.isEmpty()) {
                    name = tableName + "_" + parentTableName + "_FK";
                }

                ForeignKeyInfo fkInfo = map.get(name);
                if (fkInfo == null) {
                    fkInfo = new ForeignKeyInfo(name, foreignSchemaName, parentSchemaName, foreignTableName,
                            parentTableName);
                    map.put(name, fkInfo);
                }
                fkInfo.add(foreignColumn, parentColumnName);
            }
            return new ArrayList<ForeignKeyInfo>(map.values());
        } finally {
            rs.close();
        }
    }

    private List<ForeignKeyInfo> buildExportedKeys(DatabaseMetaData md) throws SQLException {
        ResultSet rs = md.getExportedKeys(catalog, schema, tableName);
        try {
            Map<String, ForeignKeyInfo> map = new TreeMap<>();
            while (rs.next()) {
                String name = rs.getString(FK_NAME);
                String parentSchemaName = normalizeSQLName(rs.getString(FK_PARENT_SCHEMA_NAME));
                String parentTableName = normalizeSQLName(rs.getString(FK_PARENT_TABLE_NAME));
                String parentColumnName = normalizeSQLName(rs.getString(FK_PARENT_COLUMN_NAME));
                String foreignSchemaName = normalizeSQLName(rs.getString(FK_FOREIGN_SCHEMA_NAME));
                String foreignTableName = normalizeSQLName(rs.getString(FK_FOREIGN_TABLE_NAME));
                String foreignColumn = normalizeSQLName(rs.getString(FK_FOREIGN_COLUMN_NAME));
                if (name == null || name.isEmpty()) {
                    name = tableName + "_" + foreignTableName + "_IFK";
                }

                ForeignKeyInfo fkInfo = map.get(name);
                if (fkInfo == null) {
                    fkInfo = new ForeignKeyInfo(name, foreignSchemaName, parentSchemaName, foreignTableName,
                            parentTableName);
                    map.put(name, fkInfo);
                }
                fkInfo.add(foreignColumn, parentColumnName);
            }
            return new ArrayList<ForeignKeyInfo>(map.values());
        } finally {
            rs.close();
        }
    }
    
    private void buildUniqueKeysAndIndexs(DatabaseMetaData md) throws SQLException {
        Map<String, IndexInfo> indexInfos = new TreeMap<>();
        ResultSet rs = md.getIndexInfo(catalog, schema, tableName, false, false);
        try {
            while (rs.next()) {
                String name = rs.getString("INDEX_NAME");
                boolean nonUnique = rs.getBoolean("NON_UNIQUE");
                String ascDesc = rs.getString("ASC_OR_DESC");
                String columnName = normalizeSQLName(rs.getString("COLUMN_NAME"));
                String indexSchemaName = normalizeSQLName(rs.getString("TABLE_SCHEM"));
                String indexTableName = normalizeSQLName(rs.getString("TABLE_NAME"));
                if (name == null || name.isEmpty()) {
                    continue;
                }

                IndexInfo idxInfo = indexInfos.get(name);
                if (idxInfo == null) {
                    idxInfo = new IndexInfo(name, indexSchemaName, indexTableName, !nonUnique);
                    indexInfos.put(name, idxInfo);
                }
                boolean desc = (ascDesc != null && ascDesc.equalsIgnoreCase("D"));
                idxInfo.addColumn(new IndexColumn(columnName, desc));
            }
        } finally {
            rs.close();
        }
        
        // index
        for (IndexInfo ixInfo : indexInfos.values()) {
            List<String> ixColumns = new ArrayList<>();
            for (IndexColumn column : ixInfo.getColumns()) {
                ixColumns.add(column.getColumnName());
            }
            // uniqueKey，排除因设置主键创建的
            if (ixColumns.equals(primaryKey.getColumns())) {
                continue;
            }
            // 排除因设置外键创建的
            boolean accept = true;
            for (ForeignKeyInfo fkInfo : importedKeys) {
                if (ixColumns.equals(fkInfo.getForeignColumns())) {
                    accept = false;
                    break;
                }
            }
            if (accept) {
                indexs.add(ixInfo);
            }
        }
    }

    private String normalize(String str) {
        if (lowerCase && str != null) {
            return str.toLowerCase();
        } else {
            return str;
        }
    }

    public String getTableName() {
        return tableName;
    }

    /**
     * 返回主键
     * 
     * @return
     */
    public PrimaryKeyInfo getPrimaryKey() {
        return primaryKey;
    }

    /**
     * 返回列
     * @return
     */
    public List<ColumnInfo> getColumns() {
        return columns;
    }

    /**
     * 返回外键
     * 
     * @return
     */
    public List<ForeignKeyInfo> getImportedKeys() {
        return importedKeys;
    }

    /**
     * 返回引用给定表的主键列（表导入的外键）的外键列
     * 
     * @return
     */
    public List<ForeignKeyInfo> getExportedKeys() {
        return exportedKeys;
    }

    /**
     * 返回索引
     * 
     * @return
     */
    public List<IndexInfo> getIndexs() {
        return indexs;
    }

    private static String normalizeSQLName(String name) {
        if (name != null) {
            return name.replaceAll("\r", "").replaceAll("\n", " ");
        } else {
            return null;
        }
    }

}
