/* Copyright (C) 2013-2025 TU Dortmund University
 * This file is part of AutomataLib <https://automatalib.net>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.automatalib.util.automaton.conformance;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import net.automatalib.automaton.UniversalDeterministicAutomaton;
import net.automatalib.common.util.collection.AbstractThreeLevelIterator;
import net.automatalib.common.util.collection.IterableUtil;
import net.automatalib.common.util.collection.IteratorUtil;
import net.automatalib.common.util.collection.ReusableIterator;
import net.automatalib.util.automaton.cover.Covers;
import net.automatalib.util.automaton.equivalence.CharacterizingSets;
import net.automatalib.word.Word;
import net.automatalib.word.WordBuilder;

/**
 * Iterator that returns test words generated by the W method.
 * <p>
 * See "Testing software design modeled by finite-state machines" by Tsun S. Chow.
 *
 * @param <I>
 *         input symbol type
 */
public class WMethodTestsIterator<I> extends AbstractThreeLevelIterator<Word<I>, List<I>, Word<I>, Word<I>> {

    private final Collection<? extends I> inputs;
    private final int maxDepth;

    private final Iterable<Word<I>> suffixes;

    /**
     * Convenience-constructor for {@link #WMethodTestsIterator(UniversalDeterministicAutomaton, Collection, int)} that
     * selects {@code 0} as {@code maxDepth}.
     *
     * @param automaton
     *         the automaton for which the testing sequences should be generated
     * @param inputs
     *         the input symbols that should be considered for test sequence generation
     */
    public WMethodTestsIterator(UniversalDeterministicAutomaton<?, I, ?, ?, ?> automaton,
                                Collection<? extends I> inputs) {
        this(automaton, inputs, 0);
    }

    /**
     * Constructor.
     *
     * @param automaton
     *         the automaton for which the testing sequences should be generated
     * @param inputs
     *         the input symbols that should be considered for test sequence generation
     * @param maxDepth
     *         the maximum number of symbols that are appended to the transition-cover part of the test sequences
     */
    public WMethodTestsIterator(UniversalDeterministicAutomaton<?, I, ?, ?, ?> automaton,
                                Collection<? extends I> inputs,
                                int maxDepth) {
        super(IteratorUtil.concat(IteratorUtil.singleton(Word.epsilon()),
                                  Covers.transitionCoverIterator(automaton, inputs)));

        this.inputs = inputs;
        this.maxDepth = maxDepth;

        final Iterator<Word<I>> characterizingSet = CharacterizingSets.characterizingSetIterator(automaton, inputs);

        // Special case: List of characterizing suffixes may be empty,
        // but in this case we still need to iterate over the prefixes!
        if (characterizingSet.hasNext()) {
            this.suffixes = new ReusableIterator<>(characterizingSet);
        } else {
            this.suffixes = Collections.singletonList(Word.epsilon());
        }
    }

    @Override
    protected Iterator<List<I>> l2Iterator(Word<I> l1Object) {
        return IterableUtil.<I>allTuples(inputs, 0, maxDepth).iterator();
    }

    @Override
    protected Iterator<Word<I>> l3Iterator(Word<I> prefix, List<I> middle) {
        return suffixes.iterator();
    }

    @Override
    protected Word<I> combine(Word<I> prefix, List<I> middle, Word<I> suffix) {
        final WordBuilder<I> wb = new WordBuilder<>(prefix.size() + middle.size() + suffix.size());
        return wb.append(prefix).append(middle).append(suffix).toWord();
    }
}
