/*
 * Decompiled with CFR 0.152.
 */
package net.finmath.smartcontract.valuation.service.utils;

import io.reactivex.rxjava3.functions.Consumer;
import java.io.IOException;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicReference;
import javax.xml.parsers.ParserConfigurationException;
import net.finmath.smartcontract.model.ExceptionId;
import net.finmath.smartcontract.model.InitialSettlementRequest;
import net.finmath.smartcontract.model.InitialSettlementResult;
import net.finmath.smartcontract.model.MarketDataList;
import net.finmath.smartcontract.model.RegularSettlementRequest;
import net.finmath.smartcontract.model.RegularSettlementResult;
import net.finmath.smartcontract.model.SDCException;
import net.finmath.smartcontract.model.ValueResult;
import net.finmath.smartcontract.product.SmartDerivativeContractDescriptor;
import net.finmath.smartcontract.product.xml.SDCXMLParser;
import net.finmath.smartcontract.product.xml.Smartderivativecontract;
import net.finmath.smartcontract.settlement.Settlement;
import net.finmath.smartcontract.settlement.SettlementGenerator;
import net.finmath.smartcontract.valuation.implementation.MarginCalculator;
import net.finmath.smartcontract.valuation.marketdata.data.MarketDataPoint;
import net.finmath.smartcontract.valuation.marketdata.generators.MarketDataGeneratorLauncher;
import net.finmath.smartcontract.valuation.marketdata.generators.MarketDataGeneratorScenarioList;
import net.finmath.smartcontract.valuation.service.config.RefinitivConfig;
import net.finmath.smartcontract.valuation.service.config.ValuationConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.xml.sax.SAXException;

@Service
public class SettlementService {
    private static final Logger logger = LoggerFactory.getLogger(SettlementService.class);
    private final MarginCalculator marginCalculator = new MarginCalculator();
    private final RefinitivConfig refinitivConfig;
    private final ValuationConfig valuationConfig;
    private final MarketDataGeneratorScenarioList marketDataServiceScenarioList;

    public SettlementService(RefinitivConfig refinitivConfig, ValuationConfig valuationConfig) {
        this.refinitivConfig = refinitivConfig;
        this.valuationConfig = valuationConfig;
        this.marketDataServiceScenarioList = new MarketDataGeneratorScenarioList();
    }

    public RegularSettlementResult generateRegularSettlementResult(RegularSettlementRequest regularSettlementRequest) {
        logger.info("Generating regular settlement result, liveData: {}, now parsing trade data", (Object)this.valuationConfig.isLiveMarketData());
        SmartDerivativeContractDescriptor sdc = SettlementService.parseProductData(regularSettlementRequest.getTradeData());
        logger.info("generateRegularSettlementResult - sdc trade id: {}, product marketdata provider: {}, valuation service marketdata provider: {}", new Object[]{sdc.getDltTradeId(), sdc.getMarketDataProvider(), this.valuationConfig.getLiveMarketDataProvider()});
        MarketDataList newMarketDataList = this.retrieveMarketData(sdc);
        this.includeFixingsOfLastSettlement(regularSettlementRequest, newMarketDataList);
        String newMarketDataString = SDCXMLParser.marshalClassToXMLString(newMarketDataList);
        Settlement settlementLast = SDCXMLParser.unmarshalXml(regularSettlementRequest.getSettlementLast(), Settlement.class);
        String marketDataLastString = SDCXMLParser.marshalClassToXMLString(settlementLast.getMarketData());
        logger.info("generateRegularSettlementResult - newMarketDataString: {}", (Object)newMarketDataString);
        ZonedDateTime settlementTimeNext = ZonedDateTime.now().plusDays(1L);
        ValueResult settlementValueNext = this.getValuationValueAtTime(newMarketDataString, regularSettlementRequest.getTradeData(), settlementTimeNext.toLocalDateTime());
        Map<String, BigDecimal> marginValues = this.getMargin(marketDataLastString, newMarketDataString, regularSettlementRequest.getTradeData());
        BigDecimal margin = marginValues.get("value");
        String newSettlement = new SettlementGenerator().generateRegularSettlementXml(newMarketDataString, sdc, margin).marginLimits(settlementLast.getMarginLimits()).settlementNPV(this.getValue(newMarketDataString, regularSettlementRequest.getTradeData())).settlementNPVPrevious(settlementLast.getSettlementNPV()).settlementTimeNext(settlementTimeNext).settlementNPVNext(settlementValueNext.getValue()).settlementInfo(marginValues).build();
        return new RegularSettlementResult().generatedRegularSettlement(newSettlement).currency(this.valuationConfig.getSettlementCurrency()).marginValue(margin).valuationDate(LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd-HHmmss")));
    }

    public InitialSettlementResult generateInitialSettlementResult(InitialSettlementRequest initialSettlementRequest) {
        logger.info("Generating initial settlement result, liveData: {}, now parsing trade data", (Object)this.valuationConfig.isLiveMarketData());
        SmartDerivativeContractDescriptor sdc = SettlementService.parseProductData(initialSettlementRequest.getTradeData());
        logger.info("generateInitialSettlementResult- sdc trade id: {}, product marketdata provider: {}, valuation service marketdata provider: {}", new Object[]{sdc.getDltTradeId(), sdc.getMarketDataProvider(), this.valuationConfig.getLiveMarketDataProvider()});
        MarketDataList newMarketDataList = this.retrieveMarketData(sdc);
        String newMarketDataString = SDCXMLParser.marshalClassToXMLString(newMarketDataList);
        logger.debug("generateInitialSettlementResult- newMarketDataString: {}", (Object)newMarketDataString);
        ZonedDateTime settlementTimeNext = ZonedDateTime.now().plusDays(1L);
        ValueResult settlementValueNext = this.getValuationValueAtTime(newMarketDataString, initialSettlementRequest.getTradeData(), settlementTimeNext.toLocalDateTime());
        ArrayList<BigDecimal> marginLimits = new ArrayList<BigDecimal>();
        sdc.getCounterparties().forEach(party -> marginLimits.add(BigDecimal.valueOf(sdc.getMarginAccount(party.getId()))));
        String newSettlement = new SettlementGenerator().generateInitialSettlementXml(newMarketDataString, sdc).marginLimits(marginLimits).settlementNPV(this.getValue(newMarketDataString, initialSettlementRequest.getTradeData())).settlementTimeNext(settlementTimeNext).settlementNPVNext(settlementValueNext.getValue()).settlementInfo(Map.of()).build();
        return new InitialSettlementResult().generatedInitialSettlement(newSettlement).currency(this.valuationConfig.getSettlementCurrency()).marginValue(BigDecimal.ZERO).valuationDate(LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd-HHmmss")));
    }

    private static SmartDerivativeContractDescriptor parseProductData(String tradeData) {
        try {
            return SDCXMLParser.parse(tradeData);
        }
        catch (IOException | ParserConfigurationException | SAXException e) {
            logger.error("error parsing product data ", (Throwable)e);
            throw new SDCException(ExceptionId.SDC_XML_PARSE_ERROR, "product data format incorrect, could not parse xml", 400);
        }
    }

    private MarketDataList retrieveMarketData(SmartDerivativeContractDescriptor sdc) {
        AtomicReference<MarketDataList> marketDataList = new AtomicReference<MarketDataList>(new MarketDataList());
        logger.info("retrieveMarketData started for trade: {}", (Object)sdc.getDltTradeId());
        if (sdc.getMarketDataProvider().equals(this.valuationConfig.getLiveMarketDataProvider()) && this.valuationConfig.isLiveMarketData()) {
            logger.info("using live market data provider");
            marketDataList.set(MarketDataGeneratorLauncher.instantiateMarketDataGeneratorWebsocket(this.initConnectionProperties(), sdc));
        } else if (sdc.getMarketDataProvider().equals(this.valuationConfig.getInternalMarketDataProvider())) {
            logger.info("using internal market data provider");
            Consumer marketDataWriter = marketDataList::set;
            this.marketDataServiceScenarioList.asObservable().subscribe(marketDataWriter, throwable -> logger.error("unable to generate marketData from files ", throwable), () -> logger.info("on complete, simulated marketData generated from files"));
        } else {
            logger.error("unable to retrieve marketData for {}", (Object)sdc.getDltTradeId());
            throw new SDCException(ExceptionId.SDC_WRONG_INPUT, "Product data XML is not compatible with valuation service configuration, see logs for further investigation", 400);
        }
        return marketDataList.get();
    }

    private Properties initConnectionProperties() {
        try {
            Properties connectionProperties = new Properties();
            connectionProperties.put("USER", this.refinitivConfig.getUser());
            connectionProperties.put("PASSWORD", this.refinitivConfig.getPassword());
            connectionProperties.put("CLIENTID", this.refinitivConfig.getClientId());
            connectionProperties.put("HOSTNAME", this.refinitivConfig.getHostName());
            connectionProperties.put("PORT", (Object)this.refinitivConfig.getPort());
            connectionProperties.put("AUTHURL", this.refinitivConfig.getAuthUrl());
            connectionProperties.put("USEPROXY", this.refinitivConfig.getUseProxy());
            connectionProperties.put("PROXYHOST", this.refinitivConfig.getProxyHost());
            connectionProperties.put("PROXYPORT", (Object)this.refinitivConfig.getProxyPort());
            connectionProperties.put("PROXYUSER", this.refinitivConfig.getProxyUser());
            connectionProperties.put("PROXYPASS", this.refinitivConfig.getProxyPassword());
            return connectionProperties;
        }
        catch (NullPointerException e) {
            logger.error("refinitiv connection properties not set", (Throwable)e);
            throw new SDCException(ExceptionId.SDC_NO_DATA_FOUND, "missing connection properties", 400);
        }
    }

    private ValueResult getValuationValueAtTime(String marketData, String tradeData, LocalDateTime valuationDate) {
        try {
            return this.marginCalculator.getValueAtEvaluationTime(marketData, tradeData, valuationDate);
        }
        catch (Exception e) {
            logger.error("unable to get valueAtTime for market data ", (Throwable)e);
            throw new SDCException(ExceptionId.SDC_VALUE_CALCULATION_ERROR, "error in MarginCalculator getValueAtTime");
        }
    }

    private BigDecimal getValue(String marketData, String tradeData) {
        try {
            return this.marginCalculator.getValue(marketData, tradeData).getValue();
        }
        catch (Exception e) {
            logger.error("unable to get value for market data ", (Throwable)e);
            throw new SDCException(ExceptionId.SDC_VALUE_CALCULATION_ERROR, "error in MarginCalculator getValue");
        }
    }

    private Map<String, BigDecimal> getMargin(String marketDataStart, String marketDataEnd, String tradeData) {
        try {
            return this.marginCalculator.getValues(marketDataStart, marketDataEnd, tradeData);
        }
        catch (Exception e) {
            logger.error("unable to get margin for market data ", (Throwable)e);
            throw new SDCException(ExceptionId.SDC_VALUE_CALCULATION_ERROR, "error in MarginCalculator getMargin");
        }
    }

    private void includeFixingsOfLastSettlement(RegularSettlementRequest regularSettlementRequest, MarketDataList newMarketDataList) {
        Smartderivativecontract sdc = SDCXMLParser.unmarshalXml(regularSettlementRequest.getTradeData(), Smartderivativecontract.class);
        Optional<Smartderivativecontract.Settlement.Marketdata.Marketdataitems.Item> symbolsOptional = sdc.getSettlement().getMarketdata().getMarketdataitems().getItem().stream().filter(item -> item.getType().get(0).equalsIgnoreCase(this.valuationConfig.getProductFixingType())).findAny();
        if (!symbolsOptional.isPresent()) {
            logger.warn("no Fixings found in SDC product data XML, marketDataList not changed");
            return;
        }
        List<String> symbols = symbolsOptional.get().getSymbol();
        logger.info("found symbols in product data XML: {}", symbols);
        Settlement settlementLast = SDCXMLParser.unmarshalXml(regularSettlementRequest.getSettlementLast(), Settlement.class);
        ArrayList fixingsLastSettlement = new ArrayList();
        symbols.forEach(s -> settlementLast.getMarketData().getPoints().forEach(marketDataPoint -> {
            if (marketDataPoint.getId().equalsIgnoreCase((String)s)) {
                fixingsLastSettlement.add(marketDataPoint);
            }
        }));
        logger.info("add matching marketdataPoints to product symbols: {}", fixingsLastSettlement);
        for (MarketDataPoint marketDataPoint : fixingsLastSettlement) {
            newMarketDataList.add(marketDataPoint);
        }
    }
}

