/*
 * Decompiled with CFR 0.152.
 */
package net.openhft.chronicle.queue.impl.single;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import net.openhft.chronicle.core.Jvm;
import net.openhft.chronicle.core.io.Closeable;
import net.openhft.chronicle.core.io.IOTools;
import net.openhft.chronicle.core.threads.InterruptedRuntimeException;
import net.openhft.chronicle.queue.QueueTestCommon;
import net.openhft.chronicle.queue.impl.TableStore;
import net.openhft.chronicle.queue.impl.single.TSQueueLock;
import net.openhft.chronicle.queue.impl.table.Metadata;
import net.openhft.chronicle.queue.impl.table.SingleTableBuilder;
import net.openhft.chronicle.testframework.process.ProcessRunner;
import net.openhft.chronicle.threads.Pauser;
import net.openhft.chronicle.threads.Threads;
import net.openhft.chronicle.threads.TimingPauser;
import net.openhft.chronicle.wire.UnrecoverableTimeoutException;
import org.jetbrains.annotations.NotNull;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class TSQueueLockTest
extends QueueTestCommon {
    private static final String TEST_LOCK_NAME = "testLock";
    private static final long TIMEOUT_MS = 100L;
    private TableStore<Metadata.NoMeta> tableStore;

    @Before
    public void setUp() {
        Path tempDir = IOTools.createTempDirectory((String)this.getClass().getSimpleName());
        tempDir.toFile().mkdirs();
        Path storeDirectory = tempDir.resolve("test_store.cq4t");
        this.tableStore = SingleTableBuilder.binary((Path)storeDirectory, (Metadata)Metadata.NoMeta.INSTANCE).build();
    }

    @Override
    @After
    public void tearDown() {
        Closeable.closeQuietly(this.tableStore);
    }

    @Test(timeout=5000L)
    public void lockWillThrowIllegalStateExceptionIfInterruptedWhileWaitingForLock() throws InterruptedException {
        try (TSQueueLock testLock = TSQueueLockTest.createTestLock(this.tableStore, 5000L);){
            testLock.acquireLock();
            AtomicBoolean threwException = new AtomicBoolean(false);
            Thread t = new Thread(() -> {
                try {
                    testLock.acquireLock();
                }
                catch (IllegalStateException e) {
                    threwException.set(true);
                }
            });
            t.start();
            Jvm.pause((long)10L);
            t.interrupt();
            t.join();
            Assert.assertTrue((boolean)threwException.get());
        }
    }

    @Test(timeout=5000L)
    public void testIsLockedByCurrentProcess() {
        AtomicLong actualPid = new AtomicLong(-1L);
        try (TSQueueLock testLock = this.createTestLock();){
            testLock.acquireLock();
            Assert.assertTrue((boolean)testLock.isLockedByCurrentProcess(actualPid::set));
            Assert.assertEquals((long)-1L, (long)actualPid.get());
            testLock.unlock();
            Assert.assertFalse((boolean)testLock.isLockedByCurrentProcess(actualPid::set));
            Assert.assertEquals((long)Long.MIN_VALUE, (long)actualPid.get());
        }
    }

    @Test(timeout=5000L)
    public void lockWillBeAcquiredAfterTimeoutWithAWarning() throws InterruptedException {
        try (TSQueueLock testLock = TSQueueLockTest.createTestLock(this.tableStore, 50L);){
            Thread t = new Thread(() -> ((TSQueueLock)testLock).acquireLock());
            t.start();
            t.join();
            testLock.acquireLock();
            this.expectException("Unlocking forcibly");
            this.expectException("Forced unlock");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=5000L, expected=UnrecoverableTimeoutException.class)
    public void lockWillThrowExceptionAfterTimeoutWhenDontRecoverLockTimeoutIsTrue() throws InterruptedException {
        this.expectException("queue.dont.recover.lock.timeout property is deprecated and will be removed");
        System.setProperty("queue.dont.recover.lock.timeout", "true");
        try (TSQueueLock testLock = TSQueueLockTest.createTestLock(this.tableStore, 50L);){
            Thread t = new Thread(() -> ((TSQueueLock)testLock).acquireLock());
            t.start();
            t.join();
            testLock.acquireLock();
            Assert.fail((String)"Should have thrown trying to lock()");
        }
        finally {
            System.clearProperty("queue.dont.recover.lock.timeout");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=5000L, expected=UnrecoverableTimeoutException.class)
    public void lockWillThrowExceptionAfterTimeoutWhenOnlyUnlockIfProcessDeadIsTrue() throws InterruptedException {
        System.setProperty("queue.force.unlock.mode", "LOCKING_PROCESS_DEAD");
        this.expectException("Couldn't acquire lock after");
        try (TSQueueLock testLock = TSQueueLockTest.createTestLock(this.tableStore, 50L);){
            Thread t = new Thread(() -> ((TSQueueLock)testLock).acquireLock());
            t.start();
            t.join();
            testLock.acquireLock();
            Assert.fail((String)"Should have thrown trying to lock()");
        }
        finally {
            System.clearProperty("queue.force.unlock.mode");
        }
    }

    @Test(timeout=5000L)
    public void unlockWillWarnIfNotLocked() {
        try (TSQueueLock testLock = this.createTestLock();){
            testLock.unlock();
            this.expectException("Queue lock was locked by another thread");
        }
    }

    @Test(timeout=5000L)
    public void unlockWillNotUnlockAndWarnIfLockedByAnotherProcess() throws IOException, InterruptedException, TimeoutException {
        try (TSQueueLock testLock = this.createTestLock();){
            Process process = this.runLockingProcess(true);
            this.waitForLockToBecomeLocked(testLock);
            testLock.unlock();
            Assert.assertTrue((boolean)testLock.isLocked());
            this.expectException("Queue lock was locked by another thread");
            process.destroy();
            process.waitFor();
        }
    }

    @Test(timeout=15000L)
    public void lockPreventsConcurrentAcquisition() {
        AtomicBoolean lockIsAcquired = new AtomicBoolean(false);
        try (TSQueueLock testLock = TSQueueLockTest.createTestLock(this.tableStore, 10000L);){
            int numThreads = Math.min(6, Runtime.getRuntime().availableProcessors());
            ExecutorService executorService = Executors.newFixedThreadPool(numThreads);
            CyclicBarrier barrier = new CyclicBarrier(numThreads);
            Collection futures = IntStream.range(0, numThreads).mapToObj(v -> executorService.submit(new LockAcquirer(testLock, lockIsAcquired, 30, barrier))).collect(Collectors.toList());
            futures.forEach(fut -> {
                try {
                    fut.get();
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
            Threads.shutdown((ExecutorService)executorService);
        }
    }

    @Test(timeout=5000L)
    public void forceUnlockIfProcessIsDeadWillFailWhenLockingProcessIsAlive() throws IOException, TimeoutException, InterruptedException {
        Process lockingProcess = this.runLockingProcess(true);
        try (TSQueueLock lock = this.createTestLock();){
            this.waitForLockToBecomeLocked(lock);
            Assert.assertFalse((boolean)lock.forceUnlockIfProcessIsDead());
            Assert.assertTrue((boolean)lock.isLocked());
        }
        lockingProcess.destroy();
        lockingProcess.waitFor(5000L, TimeUnit.SECONDS);
    }

    @Test(timeout=5000L)
    public void forceUnlockIfProcessIsDeadWillSucceedWhenLockingProcessIsDead() throws IOException, TimeoutException, InterruptedException {
        Process lockingProcess = this.runLockingProcess(false);
        try (TSQueueLock lock = this.createTestLock();){
            this.waitForLockToBecomeLocked(lock);
            lockingProcess.destroy();
            lockingProcess.waitFor(5000L, TimeUnit.SECONDS);
            Assert.assertTrue((boolean)lock.forceUnlockIfProcessIsDead());
            Assert.assertFalse((boolean)lock.isLocked());
        }
    }

    @Test(timeout=5000L)
    public void forceUnlockIfProcessIsDeadWillSucceedWhenLockIsNotLocked() {
        try (TSQueueLock lock = this.createTestLock();){
            Assert.assertTrue((boolean)lock.forceUnlockIfProcessIsDead());
            Assert.assertFalse((boolean)lock.isLocked());
        }
    }

    @Test
    public void forceUnlockOnBehalfOfThreadDoesNothingWhenNotLocked() {
        try (TSQueueLock lock = this.createTestLock();){
            lock.forceUnlockOnBehalfOfThread(12345L);
            Assert.assertFalse((boolean)lock.isLocked());
        }
    }

    @Test
    public void forceUnlockOnBehalfOfThreadWillSucceedWithMatchingThreadID() throws TimeoutException, InterruptedException {
        try (TSQueueLock lock = this.createTestLock();){
            Thread t = new Thread(() -> ((TSQueueLock)lock).acquireLock());
            t.start();
            this.waitForLockToBecomeLocked(lock);
            lock.forceUnlockOnBehalfOfThread(t.getId());
            Assert.assertFalse((boolean)lock.isLocked());
            t.join();
        }
    }

    @Test
    public void forceUnlockOnBehalfOfThreadWillFailWhenThreadIDDoesNotMatch() throws TimeoutException {
        this.expectException("Queue lock was locked by another thread, provided-tid");
        try (TSQueueLock lock = this.createTestLock();){
            lock.acquireLock();
            this.waitForLockToBecomeLocked(lock);
            lock.forceUnlockOnBehalfOfThread(Thread.currentThread().getId() + 1L);
            Assert.assertTrue((boolean)lock.isLocked());
        }
    }

    private void waitForLockToBecomeLocked(TSQueueLock lock) throws TimeoutException {
        TimingPauser p = Pauser.balanced();
        while (!lock.isLocked()) {
            p.pause(5000L, TimeUnit.SECONDS);
            if (!Thread.currentThread().isInterrupted()) continue;
            throw new InterruptedRuntimeException("Interrupted waiting for lock to lock");
        }
    }

    private TSQueueLock createTestLock() {
        return TSQueueLockTest.createTestLock(this.tableStore, 100L);
    }

    @NotNull
    private static TSQueueLock createTestLock(TableStore<Metadata.NoMeta> tableStore, long timeoutMilliseconds) {
        return new TSQueueLock(tableStore, Pauser::balanced, timeoutMilliseconds);
    }

    private Process runLockingProcess(boolean releaseAfterInterrupt) throws IOException {
        return ProcessRunner.runClass(LockAndHoldUntilInterrupted.class, (String[])new String[]{this.tableStore.file().getAbsolutePath(), String.valueOf(releaseAfterInterrupt)});
    }

    private static void lockAndHoldUntilInterrupted(String tableStorePath, boolean releaseWhenInterrupted) {
        try (TableStore tableStore = SingleTableBuilder.binary((String)tableStorePath, (Metadata)Metadata.NoMeta.INSTANCE).build();
             TSQueueLock lock = TSQueueLockTest.createTestLock((TableStore<Metadata.NoMeta>)tableStore, 15000L);){
            lock.acquireLock();
            while (!Thread.currentThread().isInterrupted()) {
                Jvm.pause((long)100L);
            }
            if (releaseWhenInterrupted) {
                lock.unlock();
            }
        }
    }

    static class LockAcquirer
    implements Runnable {
        private final TSQueueLock TSQueueLock;
        private final AtomicBoolean lockIsAcquired;
        private final int numberOfIterations;
        private final CyclicBarrier barrier;

        LockAcquirer(TSQueueLock TSQueueLock2, AtomicBoolean lockIsAcquired, int numberOfIterations, CyclicBarrier barrier) {
            this.TSQueueLock = TSQueueLock2;
            this.lockIsAcquired = lockIsAcquired;
            this.numberOfIterations = numberOfIterations;
            this.barrier = barrier;
        }

        @Override
        public void run() {
            try {
                this.barrier.await();
                for (int i = 0; i < this.numberOfIterations; ++i) {
                    this.TSQueueLock.acquireLock();
                    try {
                        this.lockIsAcquired.compareAndSet(false, true);
                        Jvm.pause((long)10L);
                        this.lockIsAcquired.compareAndSet(true, false);
                        continue;
                    }
                    finally {
                        this.TSQueueLock.unlock();
                        Jvm.pause((long)1L);
                    }
                }
            }
            catch (Exception e) {
                throw new AssertionError((Object)e);
            }
        }
    }

    static class LockAndHoldUntilInterrupted {
        LockAndHoldUntilInterrupted() {
        }

        public static void main(String[] args) {
            TSQueueLockTest.lockAndHoldUntilInterrupted(args[0], Boolean.parseBoolean(args[1]));
        }
    }
}

