/*
 * Copyright (C) the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sf.lucis.core;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.search.highlight.Formatter;
import org.apache.lucene.search.highlight.SimpleHTMLFormatter;

import com.google.common.base.Preconditions;

/**
 * Highlighter configuration.
 * @author Emilio Escobar Reyero
 */
public class Highlight implements Serializable {

	private static final long serialVersionUID = 2682965124967878782L;

	private final Map<String, Integer> fields;
	private final Analyzer analyzer;
	private final Formatter formatter;

	private Highlight(Analyzer analyzer, Formatter formatter, Map<String, Integer> fields) {
		this.analyzer = analyzer;
		this.formatter = formatter;
		this.fields = fields;
	}

	public static Highlight of(Analyzer analyzer, Formatter formatter, Map<String, Integer> fields) {
		Preconditions.checkNotNull(analyzer);
		Preconditions.checkNotNull(formatter);
		Preconditions.checkNotNull(fields);
		return new Highlight(analyzer, formatter, fields);
	}

	public static Highlight of(Map<String, Integer> fields) {
		Preconditions.checkNotNull(fields);
		return new Highlight(new StandardAnalyzer(), new SimpleHTMLFormatter(), fields);
	}

	public static Highlight no() {
		return new Highlight(new StandardAnalyzer(), new SimpleHTMLFormatter(), new HashMap<String, Integer>());
	}

	public Analyzer getAnalyzer() {
		return analyzer;
	}

	public Formatter getFormatter() {
		return formatter;
	}

	public Map<String, Integer> getFields() {
		return fields;
	}

}
