/*
 * Copyright (C) the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sf.lucis.core;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.List;
import java.util.Map;

import net.sf.derquinse.lucis.Group;
import net.sf.derquinse.lucis.GroupResult;
import net.sf.derquinse.lucis.Item;
import net.sf.derquinse.lucis.Page;
import net.sf.derquinse.lucis.Result;
import net.sf.lucis.core.support.AllCollector;
import net.sf.lucis.core.support.CountingCollector;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.search.Filter;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.TopDocCollector;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.highlight.Highlighter;
import org.apache.lucene.search.highlight.NullFragmenter;
import org.apache.lucene.search.highlight.QueryScorer;
import org.apache.lucene.search.highlight.SimpleFragmenter;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;

/**
 * Abstract implementation for a search service.
 * @author Andres Rodriguez
 */
public abstract class LucisQuery<T> {
	private LucisQuery() {
	}

	TopDocs getTopDocs(LucisSearcher searcher, Query query, Filter filter, Sort sort, int hits) {
		final TopDocs docs;
		if (sort == null) {
			final TopDocCollector tdc = new TopDocCollector(hits);
			searcher.search(query, filter, tdc);
			docs = tdc.topDocs();
		} else {
			docs = searcher.search(query, filter, hits, sort);
		}
		return docs;
	}

	public abstract T perform(LucisSearcher searcher);

	public static <T> LucisQuery<Item<T>> first(final Query query, final Filter filter, final Sort sort,
			final DocMapper<T> mapper, final Highlight highlight) {
		return new LucisQuery<Item<T>>() {
			public Item<T> perform(final LucisSearcher searcher) {
				final long t0 = System.currentTimeMillis();
				final TopDocs docs = getTopDocs(searcher, query, filter, sort, 1);
				final T item;
				final float score;
				if (docs.totalHits > 0) {
					final ScoreDoc sd = docs.scoreDocs[0];
					score = sd.score;
					final Document doc = searcher.doc(sd.doc);

					final Multimap<String, String> fragments;

					if (highlight != null && !highlight.getFields().isEmpty()) {

						final Query rewrote = searcher.rewrite(query);

						final Highlighter highlighter = new Highlighter(highlight.getFormatter(), new QueryScorer(rewrote));

						fragments = getFragments(highlighter, doc, highlight);
					} else {
						fragments = ArrayListMultimap.create();
					}

					item = mapper.map(sd.doc, score, doc, fragments);
				} else {
					score = 0.0f;
					item = null;
				}
				final long t1 = System.currentTimeMillis();
				return new Item<T>(docs.totalHits, score, t1 - t0, item);
			}
		};
	}

	public static <T> LucisQuery<Item<T>> first(final Query query, final Filter filter, final Sort sort,
			final DocMapper<T> mapper) {
		return first(query, filter, sort, mapper, Highlight.no());
	}

	public static <T> LucisQuery<Item<T>> first(final Query query, final Sort sort, final DocMapper<T> mapper) {
		return first(query, null, sort, mapper, Highlight.no());
	}

	public static <T> LucisQuery<Item<T>> first(final Query query, final Filter filter, final DocMapper<T> mapper) {
		return first(query, filter, null, mapper, Highlight.no());
	}

	public static <T> LucisQuery<Item<T>> first(final Query query, final DocMapper<T> mapper) {
		return first(query, null, null, mapper, Highlight.no());
	}

	public static <T> LucisQuery<Page<T>> page(final Query query, final Filter filter, final Sort sort,
			final DocMapper<T> mapper, final int first, final int pageSize, final Highlight highlight) {
		return new LucisQuery<Page<T>>() {
			public Page<T> perform(LucisSearcher searcher) {
				final int total = first + pageSize;
				final long t0 = System.currentTimeMillis();
				final TopDocs docs = getTopDocs(searcher, query, filter, sort, total);
				final List<T> items;
				final float score;
				if (docs.totalHits > first) {
					final Query rewrote = searcher.rewrite(query);

					final Highlighter highlighter = new Highlighter(highlight.getFormatter(), new QueryScorer(rewrote));

					final int n = Math.min(total, docs.scoreDocs.length);
					items = new ArrayList<T>(n - first);
					score = docs.getMaxScore();
					for (int i = first; i < n; i++) {
						final ScoreDoc sd = docs.scoreDocs[i];
						final Document doc = searcher.doc(sd.doc);

						final Multimap<String, String> fragments = getFragments(highlighter, doc, highlight);

						final T item = mapper.map(sd.doc, score, doc, fragments);
						items.add(item);
					}
				} else {
					score = 0.0f;
					items = null;
				}
				final long t1 = System.currentTimeMillis();
				return new Page<T>(docs.totalHits, score, t1 - t0, first, items);
			}
		};
	}

	private static Multimap<String, String> getFragments(final Highlighter highlighter, final Document doc,
			final Highlight highlight) {
		final Multimap<String, String> fragments = ArrayListMultimap.create();

		final Map<String, Integer> fields = highlight.getFields();
		final Analyzer analyzer = highlight.getAnalyzer();

		for (Map.Entry<String, Integer> entry : fields.entrySet()) {
			final String field = entry.getKey();
			final Integer maxNumFragments = entry.getValue();

			if (maxNumFragments >= 0) {
				final String text = doc.get(field);

				if (text != null) {
					try {
						highlighter.setTextFragmenter(maxNumFragments > 0 ? new SimpleFragmenter() : new NullFragmenter());
						String[] fr = highlighter.getBestFragments(analyzer, field, text, maxNumFragments);

						if (fr != null && fr.length > 0) {
							fragments.putAll(field, Arrays.asList(fr));
						}
					} catch (IOException e) {
					}
				}
			}
		}

		return fragments;
	}

	public static <T> LucisQuery<Page<T>> page(final Query query, final Filter filter, final Sort sort,
			final DocMapper<T> mapper, final int first, final int pageSize) {
		return page(query, filter, sort, mapper, first, pageSize, Highlight.no());
	}

	public static <T> LucisQuery<Page<T>> page(final Query query, final Sort sort, final DocMapper<T> mapper,
			final int first, final int pageSize) {
		return page(query, null, sort, mapper, first, pageSize, Highlight.no());
	}

	public static <T> LucisQuery<Page<T>> page(final Query query, final Filter filter, final DocMapper<T> mapper,
			final int first, final int pageSize) {
		return page(query, filter, null, mapper, first, pageSize, Highlight.no());
	}

	public static <T> LucisQuery<Page<T>> page(final Query query, final DocMapper<T> mapper, final int first,
			final int pageSize) {
		return page(query, null, null, mapper, first, pageSize, Highlight.no());
	}

	public static LucisQuery<Result> count(final Query query, final Filter filter) {
		return new LucisQuery<Result>() {
			@Override
			public Result perform(LucisSearcher searcher) {
				final long t0 = System.currentTimeMillis();
				final CountingCollector collector = new CountingCollector();
				searcher.search(query, filter, collector);
				final long t1 = System.currentTimeMillis();
				return new Result(collector.getCount(), collector.getMaxScore(), t1 - t0);
			}
		};
	}

	public static LucisQuery<GroupResult> group(final Query query, final Filter filter, final List<String> fields) {
		return new LucisQuery<GroupResult>() {
			@Override
			public GroupResult perform(LucisSearcher searcher) {
				final long t0 = System.currentTimeMillis();
				final AllCollector collector = new AllCollector();
				searcher.search(query, filter, collector);
				final Group g = new Group();
				final BitSet bits = collector.getBits();
				for (int i = bits.nextSetBit(0); i >= 0; i = bits.nextSetBit(i + 1)) {
					g.addHit();
					final Document d = searcher.doc(i);
					if (fields != null && !fields.isEmpty()) {
						add(g, d, 0);
					}
				}
				final long t1 = System.currentTimeMillis();
				return new GroupResult(g, collector.getMaxScore(), t1 - t0);
			}

			private void add(Group g, Document d, int i) {
				if (i < fields.size()) {
					final String f = fields.get(i);
					if (f == null) {
						return;
					}
					i++;
					String[] values = d.getValues(f);
					for (String value : values) {
						Group next = g.getGroup(value);
						next.addHit();
						add(next, d, i);
					}
				}
			}
		};
	}
}
