/*
 * Copyright (C) the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sf.lucis.core;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.index.IndexWriter.MaxFieldLength;

import com.google.common.base.Preconditions;

/**
 * Index writer configuration.
 * @author Andres Rodriguez
 */
public final class WriterConfiguration {
	private final Analyzer analyzer;
	private final MaxFieldLength maxFieldLength;

	private WriterConfiguration(final Analyzer analyzer, MaxFieldLength maxFieldLength) {
		this.analyzer = analyzer;
		this.maxFieldLength = maxFieldLength;
	}

	public Analyzer getAnalyzer() {
		return analyzer;
	}

	public MaxFieldLength getMaxFieldLength() {
		return maxFieldLength;
	}

	public static WriterConfiguration defaultConfiguration() {
		return of(new StandardAnalyzer());
	}

	public static WriterConfiguration of(Analyzer analyzer, MaxFieldLength length) {
		Preconditions.checkNotNull(analyzer, "An analyzer must be provided");
		Preconditions.checkNotNull(length, "Max field length must be provided");
		return new WriterConfiguration(analyzer, length);
	}

	public static WriterConfiguration of(Analyzer analyzer, int length) {
		return of(analyzer, new MaxFieldLength(length));
	}

	public static WriterConfiguration of(Analyzer analyzer) {
		return of(analyzer, MaxFieldLength.UNLIMITED);
	}

}
