/*
 * Copyright (C) the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sf.lucis.core.impl;

import java.util.concurrent.ScheduledExecutorService;
import java.util.logging.Level;
import java.util.logging.Logger;

import net.sf.lucis.core.Batch;
import net.sf.lucis.core.Delays;
import net.sf.lucis.core.IndexStatus;
import net.sf.lucis.core.Indexer;
import net.sf.lucis.core.IndexerService;
import net.sf.lucis.core.Store;
import net.sf.lucis.core.Writer;

/**
 * Lucene-based index manager.
 * @author Andres Rodriguez
 * @param <T> Checkpoint type.
 */
public class DefaultIndexerService<T> extends AbstractIndexService implements IndexerService {
	/** Default name. */
	private static final String NAME = "DefaultIndexerService";
	/** Writer. */
	private final Writer writer;
	/** Index store. */
	private final Store<T> store;
	/** Indexer. */
	private final Indexer<T> indexer;
	/** Delays. */
	private volatile Delays delays = Delays.constant(1000);
	/** Log. */
	private Logger log = Logger.getLogger(getClass().getName());
	/** Service name. */
	private String name = NAME;

	public DefaultIndexerService(Store<T> store, Writer writer, Indexer<T> indexer,
			ScheduledExecutorService externalExecutor, boolean pasive) {
		super(externalExecutor, pasive);
		this.store = store;
		this.writer = writer;
		this.indexer = indexer;
	}

	public DefaultIndexerService(Store<T> store, Writer writer, Indexer<T> indexer) {
		this(store, writer, indexer, null, false);
	}

	/* CONFIGURABLE PROPERTIES */

	public void setLogName(final String name) {
		this.log = Logger.getLogger(name);
	}

	public void setDelays(Delays delays) {
		this.delays = delays;
	}

	public void setName(String name) {
		this.name = (name != null && name.length() > 0) ? name : NAME;
	}

	/* END CONFIGURABLE PROPERTIES. */

	Runnable newTask() {
		return new Task();
	}

	private String format(String s) {
		return String.format("Service [%s]: %s", name, s);
	}

	private final class Task implements Runnable {
		private void interrupted() {
			log.log(Level.WARNING, format("Task interruption requested. Restoring status and exiting"));
			Thread.currentThread().interrupt();
		}

		public void run() {
			final T checkpoint;
			try {
				checkpoint = store.getCheckpoint();
			} catch (Exception e) {
				log.log(Level.SEVERE, format("Error obtaining checkpoint"), e);
				schedule(IndexStatus.ERROR, delays.getError());
				return;
			}
			final Batch<T> batch;
			try {
				batch = indexer.index(checkpoint);
			} catch (InterruptedException e) {
				interrupted();
				return;
			} catch (RuntimeException e) {
				log.log(Level.SEVERE, format("Unable to obtain batch"), e);
				schedule(delays.getError());
				return;
			}
			try {
				if (batch == null) {
					log.finest(format("Empty batch. Nothing to do."));
					schedule(delays.getIdle());
				} else {
					IndexStatus status = writer.write(store, batch);
					if (status == null) {
						log.finest(format("Writer had nothing to do."));
						schedule(delays.getIdle());
					} else {
						log.finest(format("Writing complete."));
						schedule(status, delays.getNormal());
					}
				}
			} catch (InterruptedException e) {
				interrupted();
				return;
			} catch (RuntimeException e) {
				log.log(Level.SEVERE, format("Unable to write batch"), e);
				schedule(delays.getError());
			}
		}
	}

}
