/*
 * Copyright (C) the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sf.lucis.core.impl;

import static com.google.common.base.Preconditions.checkNotNull;
import static net.sf.lucis.core.Interruption.throwIfInterrupted;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

import net.sf.lucis.core.Batch;
import net.sf.lucis.core.Batch.Addition;
import net.sf.lucis.core.IndexStatus;
import net.sf.lucis.core.Store;
import net.sf.lucis.core.Writer;
import net.sf.lucis.core.WriterConfiguration;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriter.MaxFieldLength;
import org.apache.lucene.index.Term;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.LockObtainFailedException;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;

/**
 * Default writer implementation.
 * @author Andres Rodriguez.
 */
public final class DefaultWriter implements Writer {
	/** Writer configuration. */
	private final WriterConfiguration config;
	/** Merge factor. */
	private int mergeFactor = 5;
	/** RAM Buffer Size. */
	private double ramBufferSize = 32.0;

	/** Log. */
	private Logger log = Logger.getLogger(getClass().getName());

	public DefaultWriter() {
		this(WriterConfiguration.defaultConfiguration());
	}

	public DefaultWriter(final WriterConfiguration config) {
		checkNotNull(config, "A writer configuration must be provided");
		this.config = config;
	}

	/* CONFIGURABLE PROPERTIES */

	public void setLogName(final String name) {
		this.log = Logger.getLogger(name);
	}

	public void setMergeFactor(int mergeFactor) {
		if (mergeFactor > 1) {
			this.mergeFactor = mergeFactor;
		}
	}

	public void setRamBufferSize(double ramBufferSize) {
		this.ramBufferSize = ramBufferSize;
	}

	/* END CONFIGURABLE PROPERTIES. */

	private String format(Store<?> store, String s) {
		return String.format("Default Writer[%s]: %s", store.toString(), s);
	}

	public <T> IndexStatus write(Store<T> store, Batch<T> batch) throws InterruptedException {
		Preconditions.checkNotNull(store, "A destination store must be provided.");
		if (batch == null) {
			return null;
		}
		try {
			final T oldCP = store.getCheckpoint();
			final T newCP = batch.getCheckpoint();
			if (Objects.equal(oldCP, newCP)) {
				return null;
			}
			throwIfInterrupted();
			if (!batch.isEmpty()) {
				final Analyzer analyzer = config.getAnalyzer();
				final MaxFieldLength length = config.getMaxFieldLength();
				// Check whether the index must be created
				final Directory directory = store.getDirectory();
				final boolean create = !IndexReader.indexExists(directory) || batch.isRecreate();
				final IndexWriter writer = new IndexWriter(directory, analyzer, create, length);
				writer.setMergeFactor(mergeFactor);
				writer.setRAMBufferSizeMB(ramBufferSize);
				boolean ok = false;
				try {
					// Deletions
					if (!batch.isRecreate()) {
						for (Term term : batch.getDeletions()) {
							throwIfInterrupted();
							writer.deleteDocuments(term);
						}
					}
					// Additions
					for (Addition addition : batch.getAdditions()) {
						throwIfInterrupted();
						final Analyzer aa = addition.getAnalyzer();
						writer.addDocument(addition.getDocument(), aa != null ? aa : analyzer);
					}
					// Commit
					throwIfInterrupted();
					writer.commit();
					ok = true;
					// No optimize until policy is defined.
					// writer.optimize();
				} finally {
					if (!ok) {
						rollback(writer);
					}
					writer.close();
				}
			}
			store.setCheckpoint(newCP);
			return IndexStatus.OK;
		} catch (InterruptedException ie) {
			throw ie;
		} catch (LockObtainFailedException le) {
			log.log(Level.SEVERE, format(store, "Unable to lock index"), le);
			return IndexStatus.LOCKED;
		} catch (CorruptIndexException ce) {
			log.log(Level.SEVERE, format(store, "Corrupt index"), ce);
			return IndexStatus.CORRUPT;
		} catch (IOException ioe) {
			log.log(Level.SEVERE, format(store, "I/O Error while writing"), ioe);
			return IndexStatus.IOERROR;
		} catch (Exception e) {
			log.log(Level.SEVERE, format(store, "Exception while writing"), e);
			return IndexStatus.ERROR;
		}
	}

	private void rollback(IndexWriter writer) {
		try {
			writer.rollback();
		} catch (Exception e) {
		}
	}
}
