/*
 * Copyright (C) the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sf.lucis.core.impl;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import net.sf.derquinse.lucis.IndexNotAvailableException;
import net.sf.lucis.core.DirectoryProvider;
import net.sf.lucis.core.LucisSearcher;

import org.apache.lucene.index.IndexReader;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.MultiSearcher;
import org.apache.lucene.search.Searcher;
import org.apache.lucene.store.Directory;

import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;

/**
 * MultiSearcher simple implementation.
 * @author Emilio Escobar Reyero
 */
public abstract class MultiSearcherProvider implements Supplier<LucisSearcher> {

	public static MultiSearcherProvider of(final Collection<DirectoryProvider> providers) {
		Preconditions.checkNotNull(providers);
		return new MultiSearcherProvider() {
			@Override
			Collection<Directory> directories() {
				Collection<Directory> directories = new HashSet<Directory>();

				for (DirectoryProvider dp : providers) {
					directories.add(dp.getDirectory());
				}

				return ImmutableSet.copyOf(directories);
			}
		};
	}

	private MultiSearcherProvider() {
	}

	abstract Collection<Directory> directories();

	public LucisSearcher get() {
		try {
			Collection<Directory> dirs = directories();
			List<IndexSearcher> searchers = Lists.newArrayListWithCapacity(dirs.size());
			for (Directory d : dirs) {
				if (IndexReader.indexExists(d)) {
					final IndexReader reader = IndexReader.open(d, true);
					final IndexSearcher searcher = new IndexSearcher(reader);
					searchers.add(searcher);
				}
			}
			if (searchers.isEmpty()) {
				final IndexReader reader = IndexReader.open(EmptyDirectory.get(), true);
				final IndexSearcher searcher = new IndexSearcher(reader);
				searchers.add(searcher);
			}
			return new MultipleSearcher(searchers.toArray(new IndexSearcher[searchers.size()]));
		} catch (Exception e) {
			throw new IndexNotAvailableException(e);
		}
	}

	private static final class MultipleSearcher extends AbstractSearcher {
		private MultiSearcher searcher;

		private MultipleSearcher(final IndexSearcher[] searchers) {
			try {
				this.searcher = new MultiSearcher(searchers);
			} catch (IOException e) {
				this.searcher = null;
			}
		}

		@Override
		Searcher searcher() {
			return searcher;
		}

		public void close() {
			try {
				searcher.close();
			} catch (Exception e) {
			}
		}
	}

}
