/*
 * Copyright (C) the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sf.lucis.core.support;

import org.apache.lucene.search.HitCollector;

/**
 * A hit collector that simply counts the number of hits.
 * @author Andres Rodriguez
 */
public class CountingCollector extends HitCollector {
	private float maxScore;
	private int count;

	/** Constructor. */
	public CountingCollector() {
		reset();
	}

	/*
	 * (non-Javadoc)
	 * @see org.apache.lucene.search.HitCollector#collect(int, float)
	 */
	@Override
	public void collect(int doc, float score) {
		count++;
		maxScore = Math.max(maxScore, score);
	}

	/** Resets the collector. */
	public void reset() {
		maxScore = Float.MIN_VALUE;
		count = 0;
	}

	/**
	 * Returns the document count.
	 * @return The document count
	 */
	public final int getCount() {
		return count;
	}

	/**
	 * Returns the maximum collected score.
	 * @return The maximum collected score.
	 */
	public final float getMaxScore() {
		return maxScore;
	}
}
