/*
 * $Id: Resources.java 375 2012-04-25 07:30:56Z tweea $
 * Copyright(C) 2008 Matrix
 * All right reserved.
 */
package net.matrix.text;

import java.util.HashMap;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 读取多语言资源文件。
 */
public final class Resources {
	/**
	 * 日志记录器。
	 */
	private static final Logger LOG = LoggerFactory.getLogger(Resources.class);

	/**
	 * 缓存。
	 */
	private static final Map<String, Resources> RESOURCES = new HashMap<String, Resources>();

	/**
	 * 关联的资源文件。
	 */
	private ResourceBundle bundle;

	/**
	 * 使用资源文件位置实例化。
	 * 
	 * @param key
	 *            资源文件位置
	 */
	private Resources(final String key) {
		try {
			bundle = ResourceBundle.getBundle(key);
		} catch (MissingResourceException e) {
			LOG.warn(key + " 资源加载失败", e);
		}
	}

	/**
	 * 根据位置加载资源。
	 * 
	 * @param key
	 *            位置
	 * @return 资源
	 */
	public static Resources getResources(final String key) {
		Resources res = RESOURCES.get(key);
		if (res != null) {
			return res;
		}
		synchronized (RESOURCES) {
			res = RESOURCES.get(key);
			if (res == null) {
				res = new Resources(key);
				RESOURCES.put(key, res);
			}
		}
		return res;
	}

	/**
	 * 获取多语言字符串，如果失败直接返回名字。
	 * 
	 * @param name
	 *            字符串名
	 * @return 字符串
	 */
	public String getProperty(final String name) {
		if (bundle == null) {
			return name;
		}
		try {
			return bundle.getString(name);
		} catch (MissingResourceException e) {
			LOG.warn("找不到名为 " + name + " 的资源项");
			return name;
		}
	}

	/**
	 * 加载资源并返回其中指定的字符串。
	 * 
	 * @param key
	 *            位置
	 * @param name
	 *            字符串名
	 * @return 字符串
	 */
	public static String getProperty(final String key, final String name) {
		return getResources(key).getProperty(name);
	}
}
