package net.sf.sido.array;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.common.primitives.Ints;
import net.sf.sido.common.AnnotatedSupport;

/**
 * The {@link Array} class represents an immutable and ordered collection. The only ways to update
 * such an {@link Array} is to create a new one, using operations like {@link #add(Array)},
 * {@link #delete(int...)}, etc. that return other instances of arrays.
 * <p>
 * When an {@link Array} is created, it is associated with an {@link ArrayEvent} that defines the
 * list of events that have occured on a source to create this new array.
 * 
 * @author Damien Coraboeuf
 * 
 * @param <T>
 *            Type of element in the array
 */
public class Array<T> extends AnnotatedSupport implements Iterable<T>, Serializable {

	/**
	 * Version number
	 */
	private static final long serialVersionUID = 1;

	/**
	 * Creates an empty array.
	 * 
	 * @param <V>
	 *            Type of element
	 * @return Empty and invariant array
	 */
	public static <V> Array<V> empty() {
		List<V> list = Collections.emptyList();
		return new Array<V>(list);
	}

	// Underlying list
	private final List<T> list;

	// Creation event
	private final ArrayEvent<T> arrayEvent;

	/**
	 * Constructor from a list or collection
	 * 
	 * @param elements
	 *            List of elements to build the array from. The final array is disconnected from the
	 *            initial list: modifying the list won't have any impact on the array.
	 */
	public Array(Collection<T> elements) {
		this(elements, null);
	}

	/**
	 * Protected constructor for event
	 */
	protected Array(Collection<T> elements, ArrayEvent<T> event) {
		this.list = ImmutableList.copyOf(elements);
		if (event == null) {
			// Creation
			event = ArrayEvent.from(this.list, ArrayEventType.ADDED, Range.indexes(this.list));
		}
		this.arrayEvent = event;
	}

	/**
	 * Constructor from an array of elements
	 * 
	 * @param elements
	 *            Array of elements to build this {@link Array} from.
	 */
	public Array(T... elements) {
		this(Arrays.asList(elements), null);
	}

	/**
	 * Appends elements
	 * 
	 * @param elements
	 *            {@link Array} of elements to add
	 * @return A new {@link Array}, that contains this array's elements, plus the ones in
	 *         <code>elements</code>
	 */
	public Array<T> add(Array<T> elements) {
		return add(elements.toList());
	}

	/**
	 * Appends elements
	 * 
	 * @param elements
	 *            Collection of elements to add
	 * @return A new {@link Array}, that contains this array's elements, plus the ones in
	 *         <code>elements</code>
	 */
	public Array<T> add(Collection<T> elements) {
		return insert(length(), elements);
	}

	/**
	 * Appends elements
	 * 
	 * @param elements
	 *            Array of elements to add
	 * @return A new {@link Array}, that contains this array's elements, plus the ones in
	 *         <code>elements</code>
	 */
	public Array<T> add(T... elements) {
		return add(Arrays.asList(elements));
	}

    /**
     * Removes an element from the array.
     * @param element Element to remove
     * @return New array or this one if the element was not found
     */
    public Array<T> remove (T element) {
        int position = indexOf(element);
        if (position < 0) {
            return this;
        } else {
            return delete(position);
        }
    }

	/**
	 * Deletes one or several elements using their indexes. Note that the <code>indexes</code> array
	 * does not need to be sorted.
	 * 
	 * @param indexes
	 *            List of indexes to remove from the array.
	 * @return A new {@link Array} the elements have been removed from. This new array will be
	 *         associated with an {@link ArrayEvent} that describes the deletion.
	 */
	public Array<T> delete(int... indexes) {
		// Sorts indexes by increasing order
		Arrays.sort(indexes);
		// Event to generate
		ArrayEvent<T> event = new ArrayEvent<T>();
		// New list
		List<T> newList = new ArrayList<T>(this.list);
		// For all indexes to delete, from the highest to the lowest
		for (int i = indexes.length - 1; i >= 0; i--) {
			int index = indexes[i];
			if (index >= 0 && index < this.list.size()) {
				T removed = newList.remove(index);
				if (removed != null) {
					event.add(index, ArrayEventType.DELETED, removed);
				}
			}
		}
		// Resulting array
		return createNewArray(newList, event);
	}

	@SuppressWarnings("rawtypes")
	@Override
	public boolean equals(Object o) {
		if (o instanceof Array) {
			return list.equals(((Array) o).list);
		} else {
			return false;
		}
	}

    /**
     * Gets the position of an element in the array
     * @param element Element to get the position of
     * @return Position in the array, starting from 0. Returns -1 if not found.
     */
    public int indexOf (T element) {
        int position = list.indexOf(element);
        return position < 0 ? -1 : position;
    }

	/**
	 * Filters the array using a predicate
	 * 
	 * @param predicate
	 *            Function that returns <code>true</code> for the elements that must be kept
	 * @return A new {@link Array} the elements have been removed from. This new array will be
	 *         associated with an {@link ArrayEvent} that describes the deletion.
	 */
	public Array<T> filter(Predicate<T> predicate) {
		List<Integer> removedIndexes = new ArrayList<Integer>();
		List<T> newList = new ArrayList<T>(this.list);
		int index = 0;
		for (Iterator<T> i = newList.iterator(); i.hasNext();) {
			T element = i.next();
			if (!predicate.apply(element)) {
				i.remove();
				removedIndexes.add(index);
			}
			index++;
		}
		ArrayEvent<T> event = ArrayEvent.from(this.list, ArrayEventType.DELETED, Ints.toArray(removedIndexes));

		return createNewArray(newList, event);
	}

	/**
	 * Gets an element using its index
	 * 
	 * @param index
	 *            Index of the element to get
	 * @return Element at this index
	 * @throws IndexOutOfBoundsException
	 *             If the index is not valid
	 * @see List#get(int)
	 */
	public T get(int index) {
		return this.list.get(index);
	}

	/**
	 * Gets the {@link ArrayEvent} associated with this {@link Array}.
	 * 
	 * @return {@link ArrayEvent} that describes the changes that have created this {@link Array}
	 */
	public ArrayEvent<T> getArrayEvent() {
		return this.arrayEvent;
	}

	/**
	 * Same method than {@link #get(int)} but is useable for script engines like Groovy.
	 * 
	 * @param index
	 *            Index of the element to get
	 * @return Element at this index
	 * @throws IllegalArgumentException
	 *             If <code>index</code> is null
	 * @see #get(int)
	 */
	public T getAt(Integer index) {
		if (index == null) {
			throw new IllegalArgumentException("index must be not null");
		} else {
			return get(index);
		}
	}

	/**
	 * Length of this array
	 * 
	 * @return Length
	 * @see List#size()
	 * @see #length()
	 */
	public int getLength() {
		return this.list.size();
	}

    /**
     * Length of this array.
     *
     * @return Length
     * @see #length()
     */
    public int size() {
        return getLength();
    }

	/**
	 * Gets an element using its index and returns <code>null</code> if the index is not available.
	 * It still returns an {@link IndexOutOfBoundsException} if the index is less than 0.
	 * 
	 * @param index
	 *            Index of the element
	 * @return An element or <code>null</code>
	 */
	public T getOrNull(int index) {
		if (index >= list.size()) {
			return null;
		} else {
			return get(index);
		}
	}

	@Override
	public int hashCode() {
		return list.hashCode();
	}

	/**
	 * Returns an immutable index of the array using an indexing function.
	 * 
	 * @param indexer
	 *            Function that extracts the index key from the elements in the array
	 * @param <K>
	 *            Type for the index key
	 * @return Immutable map that indexes the elements of the array
	 */
	public <K> Map<K, T> index(Function<T, K> indexer) {
		return Maps.uniqueIndex(this, indexer);
	}

	/**
	 * Appends elements at a given position
	 * 
	 * @param position
	 *            Position in the array to insert the new elements at
	 * @param elements
	 *            List of elements to insert at this position
	 * @return A new {@link Array} where the elements have been inserted into. This new array will
	 *         be associated with an {@link ArrayEvent} that describes the insertion.
	 */
	public Array<T> insert(int position, Collection<T> elements) {
		List<T> newList = new ArrayList<T>(this.list);
		newList.addAll(position, elements);
		ArrayEvent<T> event = new ArrayEvent<T>();
		int index = 0;
		for (T element : elements) {
			event.add(index + position, ArrayEventType.ADDED, element);
			index++;
		}
		return createNewArray(newList, event);
	}

    /**
     * Creates a new instance of an array
     * @param newList List of elements
     * @param event Associated array of events
     * @return New instance
     */
    protected Array<T> createNewArray(List<T> newList, ArrayEvent<T> event) {
        return new Array<T>(newList, event);
    }

    /**
	 * Appends elements at a given position
	 * 
	 * @param position
	 *            Position in the array to insert the new elements at
	 * @param items
	 *            List of elements to insert at this position
	 * @return A new {@link Array} where the elements have been inserted into. This new array will
	 *         be associated with an {@link ArrayEvent} that describes the insertion.
	 */
	public Array<T> insert(int position, T... items) {
		return insert(position, Arrays.asList(items));
	}

	/**
	 * Checks if this array is empty
	 * 
	 * @return <code>true</code> is the array is empty, <code>false</code> otherwise
	 * @see List#isEmpty()
	 */
	public boolean isEmpty() {
		return this.list.isEmpty();
	}

	@Override
	public Iterator<T> iterator() {
		return this.list.iterator();
	}

	/**
	 * Length (short version) - synonym for {@link #getLength()}
	 * 
	 * @return Size of the array
	 * @see #getLength()
	 */
	public int length() {
		return getLength();
	}

	/**
	 * Sub array using a range between two indexes. If the indexes are outside of the array
	 * boundaries, they adjusted to fit in.
	 * 
	 * @param startIndex
	 *            Start index
	 * @param endIndex
	 *            End index
	 * @return A new {@link Array} instance that comprises only the elements between
	 *         <code>startIndex</code> and <code>endIndex</code>. An {@link ArrayEvent} is
	 *         associated with this array, which describes the operation.
	 * @see List#subList(int, int)
	 */
	public Array<T> sub(int startIndex, int endIndex) {
		int len = this.list.size();
		// Checks the boundaries
		if (startIndex < 0) {
			startIndex = 0;
		}
		if (endIndex > len) {
			endIndex = len;
		}
		if (startIndex > endIndex) {
			startIndex = endIndex;
		}
		// Sub
		List<T> newList = new ArrayList<T>(this.list);
		newList = newList.subList(startIndex, endIndex);
		List<Integer> removedIndexes = new ArrayList<Integer>();
		if (startIndex > 0) {
			removedIndexes.addAll(Ints.asList(new Range(0, startIndex - 1).toIndexes()));
		}
		if (endIndex < len) {
			removedIndexes.addAll(Ints.asList(new Range(endIndex, len - 1).toIndexes()));
		}
		return createNewArray(newList, ArrayEvent.from(this.list, ArrayEventType.DELETED, Ints.toArray(removedIndexes)));
	}

	/**
	 * Returns a list that contains all the elements of this array and in the same order.
	 * 
	 * @return Immutable list
	 */
	public List<T> toList() {
		return this.list;
	}

	/**
	 * String representation.
	 * 
	 * @see List#toString()
	 */
	@Override
	public String toString() {
		return this.list.toString();
	}

	/**
	 * Transforms an array in another array.
	 * 
	 * @param transform
	 *            Function that transforms each array element
	 * @param <V>
	 *            Type of the target element
	 * @return {@link Array} instance that contains a transformed version of each element of this
	 *         array
	 */
	public <V> Array<V> transform(Function<? super T, V> transform) {
		return new Array<V>(Collections2.transform(this.list, transform));
	}

	/**
	 * Updates an element with a new value
	 * 
	 * @param index
	 *            Index to update the element at
	 * @param newElement
	 *            Element to replace the initial element with
	 * @return A new {@link Array} where the elements have been updated. This new array will be
	 *         associated with an {@link ArrayEvent} that describes the update.
	 */
	public Array<T> update(int index, T newElement) {
		List<T> newList = new ArrayList<T>(this.list);
		newList.set(index, newElement);
		return createNewArray(newList, new ArrayEvent<T>().add(index, ArrayEventType.UPDATED, newElement));
	}

    /**
     * Creates a reversed version of this array
     * @return Reversed array
     * @see Collections#reverse(java.util.List) 
     */
    public Array<T> reverse() {
        // New reversed list
        List<T> reversedList = new ArrayList<T> (list);
        Collections.reverse(reversedList);
        // Creates the new array
        return createNewArray(reversedList, null);
    }

}
