package net.sf.sido.array;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Maps;
import com.google.common.primitives.Ints;

import java.util.*;

/**
 * {@link Array} sub-class that indexes its elements using an indexer function
 *
 * @param <T> Type of elements in the array
 * @param <K> Type of index
 */
public final class IndexedArray<T, K> extends Array<T> {

    private final Function<? super T, K> indexer;

    // Underlying index
    private final Map<K, T> index;

    /**
     * Builds an indexed array from another array.
     *
     * @param indexer  Indexer to use
     * @param elements Source array
     */
    public IndexedArray(Function<? super T, K> indexer, Array<T> elements) {
        super(elements.toList());
        this.indexer = indexer;
        this.index = Maps.uniqueIndex(this, indexer);
    }

    /**
     * Builds an indexed array from a collection.
     *
     * @param indexer  Indexer to use
     * @param elements Source collection
     */
    public IndexedArray(Function<? super T, K> indexer, Collection<T> elements) {
        super(elements);
        this.indexer = indexer;
        this.index = Maps.uniqueIndex(this, indexer);
    }

    /**
     * Builds an indexed array from an array.
     *
     * @param indexer  Indexer to use
     * @param elements Source array
     */
    public IndexedArray(Function<? super T, K> indexer, T... elements) {
        super(elements);
        this.indexer = indexer;
        this.index = Maps.uniqueIndex(this, indexer);
    }

    /**
     * Internal constructor used to create indexed arrays with their array events.
     *
     * @param elements Source elements
     * @param event    Array event that created this array
     * @param indexer  Indexer to use
     */
    protected IndexedArray(Collection<T> elements, ArrayEvent<T> event, Function<? super T, K> indexer) {
        super(elements, event);
        this.indexer = indexer;
        this.index = Maps.uniqueIndex(this, indexer);
    }

    /**
     * Returns the indexer function which is used by the indexed array.
     *
     * @return Indexer function
     */
    public Function<? super T, K> getIndexer() {
        return indexer;
    }

    /**
     * Gets an element by its key
     *
     * @param key Key
     * @return Associated item (can be <code>null</code>)
     */
    public T getByIndex(K key) {
        return index.get(key);
    }

    /**
     * Returns all keys as an array
     *
     * @return Array
     */
    public Array<K> getKeyArray() {
        return new Array<K>(index.keySet());
    }

    /**
     * Returns all keys as a set
     *
     * @return Set of keys
     */
    public Set<K> getKeys() {
        return index.keySet();
    }

    /**
     * Returns a list of keys, ordered the same way of the items.
     *
     * @return List of keys
     */
    public List<K> getKeyList() {
        return new ArrayList<K>(index.keySet());
    }

    /**
     * Filter this array using a predicate on the keys
     *
     * @param predicate Predicate to use on the keys
     * @return Filtered indexed array
     */
    public IndexedArray<T, K> filterOnKeys(Predicate<K> predicate) {
        List<Integer> removedIndexes = new ArrayList<Integer>();
        List<K> keyList = getKeyList();
        List<T> initialList = toList();
        List<T> newList = new ArrayList<T>(initialList);
        int index = 0;
        for (Iterator<T> i = newList.iterator(); i.hasNext();) {
            T element = i.next();
            K key = keyList.get(index);
            if (!predicate.apply(key)) {
                i.remove();
                removedIndexes.add(index);
            }
            index++;
        }
        ArrayEvent<T> event = ArrayEvent.from(initialList, ArrayEventType.DELETED, Ints.toArray(removedIndexes));

        return new IndexedArray<T, K>(newList, event, indexer);
    }

    /**
     * Transforms this indexed array by transforming the values and re-indexing them using a new indexer function.
     *
     * @param valueTransform Transform function for the values
     * @param newIndexer     Indexer for the new values
     * @param <V>            Type of the new values
     * @param <L>            Type of the new keys
     * @return Transformed indexed array
     */
    public <V, L> IndexedArray<V, L> transform(Function<? super T, V> valueTransform, Function<? super V, L> newIndexer) {
        // Array transformation
        Array<V> newValues = transform(valueTransform);
        // Indexation
        return new IndexedArray<V, L>(newIndexer, newValues);
    }

    /**
     * Creation of an IndexedArray by update/add/delete operations.
     */
    @Override
    protected Array<T> createNewArray(List<T> newList, ArrayEvent<T> tArrayEvent) {
        return new IndexedArray<T, K>(newList, tArrayEvent, indexer);
    }

    // Array functions and cast

    @Override
    public IndexedArray<T, K> add(Array<T> elements) {
        checkKeys(elements.toList());
        return (IndexedArray<T, K>) super.add(elements);
    }

    @Override
    public IndexedArray<T, K> add(Collection<T> elements) {
        checkKeys(elements);
        return (IndexedArray<T, K>) super.add(elements);
    }

    @Override
    public IndexedArray<T, K> add(T... elements) {
        checkKeys(Arrays.asList(elements));
        return (IndexedArray<T, K>) super.add(elements);
    }

    @Override
    public IndexedArray<T, K> delete(int... indexes) {
        return (IndexedArray<T, K>) super.delete(indexes);
    }

    @Override
    public IndexedArray<T, K> filter(Predicate<T> tPredicate) {
        return (IndexedArray<T, K>) super.filter(tPredicate);
    }

    @Override
    public IndexedArray<T, K> insert(int position, T... items) {
        checkKeys(Arrays.asList(items));
        return (IndexedArray<T, K>) super.insert(position, items);
    }

    @Override
    public IndexedArray<T, K> sub(int startIndex, int endIndex) {
        return (IndexedArray<T, K>) super.sub(startIndex, endIndex);
    }

    @Override
    public IndexedArray<T, K> update(int index, T newElement) {
        K oldKey = indexer.apply(get(index));
        K newKey = indexer.apply(newElement);
        if (!newKey.equals(oldKey) && this.index.containsKey(newKey)) {
            throw new IllegalArgumentException(String.format("An indexed array cannot contain duplicate keys: %s", newKey));
        }
        return (IndexedArray<T, K>) super.update(index, newElement);
    }

    @Override
    public IndexedArray<T, K> reverse() {
        return (IndexedArray<T, K>) super.reverse();
    }

    @Override
    public IndexedArray<T, K> insert(int position, Collection<T> elements) {
        checkKeys(elements);
        return (IndexedArray<T, K>) super.insert(position, elements);
    }

    /**
     * Checks that keys are unique
     */
    protected void checkKeys(Collection<T> elements) {
        // Existing set of keys
        Set<K> keys = new HashSet<K>(index.keySet());
        // Loops through all elements
        for (T element : elements) {
            // Gets the key
            K e = indexer.apply(element);
            if (keys.contains(e)) {
                throw new IllegalArgumentException(String.format("An indexed array cannot contain duplicate keys: %s", e));
            } else {
                keys.add(e);
            }
        }
    }

    @Override
    public String toString() {
        return index.toString();
    }
}
