package net.sf.sido.array;

import com.google.common.base.Function;
import com.google.common.primitives.Ints;

import java.util.Collection;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * The <code>Range</code> class defines a range between two integers. It can also
 * be a singleton (single-value range) or empty (<code>nil</code>). A <code>Range</code>
 * is immutable.
 */
public final class Range {

    /**
     * Regular expression pattern used to parse a <code>Range</code> from
     * a <code>String</code>.
     *
     * @see #valueOf(String)
     */
    public static final Pattern REGEX = Pattern
            .compile("^\\[(\\d+)(\\.\\.(\\d+))?\\]$");

    /**
     * Empty range
     */
    public static final Range NIL = new Range();

    /**
     * Creates a range from the boundaries of a collection. The
     * range will span from 0 to the size of the collection minus 1.
     * An empty or <code>null</code> collection will return the <code>nil</code>
     * range.
     *
     * @param collection Reference collection
     * @return Range that spans the size of the collection
     */
    public static Range from(Collection<?> collection) {
        if (collection == null || collection.isEmpty()) {
            return NIL;
        } else {
            return new Range(0, collection.size() - 1);
        }
    }

    /**
     * Creates a range from the boundaries of a collection, and offsets it. The
     * range will span from offset to the size of the collection minus 1 plus offset.
     * An empty or <code>null</code> collection will return the <code>nil</code>
     * range.
     *
     * @param offset     Offset to apply to the range
     * @param collection Reference collection
     * @return Range that spans the size of the collection
     * @see #from(java.util.Collection)
     */
    public static Range from(int offset, Collection<?> collection) {
        return from(collection).offset(offset);
    }

    /**
     * Gets a range from a collection and returns the corresponding
     * indexes.
     *
     * @param collection Reference collection
     * @return A list of indexes
     * @see #from(java.util.Collection)
     * @see #toIndexes()
     */
    public static int[] indexes(Collection<?> collection) {
        return Range.from(collection).toIndexes();
    }

    /**
     * Gets a range from a collection, offsets it and returns the corresponding
     * indexes.
     *
     * @param offset     Offset to apply
     * @param collection Reference collection
     * @return A list of indexes
     * @see #from(int, java.util.Collection)
     * @see #toIndexes()
     */
    public static int[] indexes(int offset, Collection<?> collection) {
        return Range.from(offset, collection).toIndexes();
    }

    /**
     * Returns the <code>nil</code> range
     *
     * @return <code>nil</code> range
     */
    public static Range nil() {
        return NIL;
    }

    /**
     * Utility method that creates an <code>Array</code> from a range.
     *
     * @param from From boundary
     * @param to   To boundary
     * @return An Array of Integer
     * @see #Range(int, int)
     * @see #toArray()
     */
    public static Array<Integer> toArray(int from, int to) {
        return new Range(from, to).toArray();
    }

    /**
     * Utility method that creates an <code>Array</code> from a range.
     *
     * @param from    From boundary
     * @param to      To boundary
     * @param creator Creates the item from an integer
     * @return An Array
     * @see #Range(int, int)
     * @see #toArray()
     */
    public static <T> Array<T> toArray(int from, int to, Function<Integer, T> creator) {
        return new Array<T>(toList(from, to, creator));
    }

    /**
     * Parses a <code>Range</code> from a <code>String</code>. The input
     * string must be formatted like:
     * <pre>
     * [from{..to}]
     * </pre>
     * If the input string is <code>null</code>, empty or equal to "[]", the
     * returned range is <code>nil</code>.
     *
     * @param value Input string
     * @return Range
     */
    public static Range valueOf(String value) {
        if (value == null || value.trim().length() == 0 || "[]".equals(value)) {
            return NIL;
        } else {
            Matcher matcher = REGEX.matcher(value);
            if (matcher.matches()) {
                String fromValue = matcher.group(1);
                int from = Integer.parseInt(fromValue, 10);
                String toValue = matcher.group(3);
                if (toValue != null) {
                    int to = Integer.parseInt(toValue, 10);
                    return new Range(from, to);
                } else {
                    return new Range(from, from);
                }
            } else {
                throw new IllegalArgumentException("Wrong range expression: "
                        + value);
            }
        }
    }

    private final boolean nil;

    private final int from;

    private final int to;

    /**
     * <code>nil</code> constructor
     */
    private Range() {
        this.nil = true;
        this.from = this.to = 0;
    }

    /**
     * Singleton constructor
     */
    public Range(int value) {
        this(value, value);
    }

    /**
     * Constructor using two boundaries.
     *
     * @param from Minimum boundary
     * @param to   Maximum boundary
     * @throws IllegalArgumentException If from is greater than to
     */
    public Range(int from, int to) {
        if (from > to) {
            throw new IllegalArgumentException("From must be <= to");
        }
        this.nil = false;
        this.from = from;
        this.to = to;
    }

    /**
     * Tests if this range contains a given value
     *
     * @param index Value to test
     * @return <code>true</code> if <code>index</code> is included
     *         into this range, <code>false</code> otherwise.
     */
    public boolean contains(int index) {
        return !this.nil && index >= this.from && index <= this.to;
    }

    /**
     * Compares two ranges. Two ranges are equal if they are <i>both</i> not <code>null</code> and if
     * their boundaries are equal.
     *
     * @param obj Object to compare against
     * @return Result of the comparison
     */
    @Override
    public boolean equals(Object obj) {
        if (obj instanceof Range) {
            Range that = (Range) obj;
            if (this.isNil() || that.isNil()) {
                // Nil ranges are never equivalent, even to a nil one
                return false;
            } else {
                // Compares the boundaries
                return (this.from == that.from) && (this.to == that.to);
            }
        } else {
            return false;
        }
    }

    /**
     * Returns a range that has been expanded to include the given target. If this range is <code>nil</code>,
     * the returned range will be the single-value range that includes the target. If this range contains the target,
     * it is returned unchanged.
     *
     * @param target Target to expand to
     * @return Expanded range
     */
    public Range expand(int target) {
        if (isNil()) {
            return new Range(target, target);
        } else if (target >= this.from && target <= this.to) {
            // No change
            return this;
        } else {
            int newFrom = Math.min(this.from, this.to);
            int newTo = Math.max(this.to, target);
            return new Range(newFrom, newTo);
        }
    }

    /**
     * Returns the minimum boundary. It is 0 for a <code>nil</code> range.
     *
     * @return Minimum boundary
     */
    public int getFrom() {
        return this.from;
    }

    /**
     * Returns the maximum boundary. It is 0 for a <code>nil</code> range.
     *
     * @return Maximum boundary
     */
    public int getTo() {
        return this.to;
    }

    /**
     * The hash code is computed from the boundaries
     *
     * @return Hash code
     */
    @Override
    public int hashCode() {
        return this.from + this.to;
    }

    /**
     * Checks if this range is <code>nil</code> or not.
     *
     * @return <code>true</code> if this range is <code>nil</code>, <code>false</code> otherwise.
     */
    public boolean isNil() {
        return this.nil;
    }

    /**
     * Checks if this range is a singleton. It is a singleton if it is not <code>nil</code> and if the two
     * boundaries are equal to each other.
     *
     * @return Singleton
     */
    public boolean isSingleton() {
        return !this.nil && (this.from == this.to);
    }

    /**
     * Creates a new range that is offset from this range.
     * <p/>
     * A <code>nil</code> range is returned unchanged.
     * <p/>
     * Examples:
     * <pre>
     * new Range(0,5).offset(0) ==> [0,5]
     * new Range(0,5).offset(2) ==> [2,7]
     * new Range(0,5).offset(-2) ==> [-2,3]
     * </pre>
     *
     * @param offset
     * @return
     */
    public Range offset(int offset) {
        if (this.nil) {
            // No change
            return this;
        } else {
            return new Range(offset + this.from, offset + this.to);
        }
    }

    /**
     * Returns the size of this range. A <code>nil</code> range has a size of zero.
     * <p/>
     * Examples:
     * <pre>
     * Range.nil().size() ==> 0
     * new Range(1).size() ==> 1
     * new Range(1,2).size() ==> 2
     * new Range(1,10).size() ==> 10
     * new Range(5,15).size() ==> 11
     * </pre>
     *
     * @return
     */
    public int size() {
        if (isNil()) {
            return 0;
        } else {
            return this.to - this.from + 1;
        }
    }

    /**
     * Converts this range into an Array of Integer that contains all the indexes defined by this range.
     * A <code>nil</code> range is converted into an empty Array.
     *
     * @return An Array of Integer
     */
    public Array<Integer> toArray() {
        if (isNil()) {
            return new Array<Integer>();
        } else {
            return new Array<Integer>(Ints.asList(toIndexes()));
        }
    }

    /**
     * Returns the list of integers included in this range.
     * Examples:
     * <pre>
     * Range.nil().toIndexes() ==> []
     * new Range(1).toIndexes() ==> [1]
     * new Range(1,2).toIndexes() ==> [1,2]
     * new Range(5,10).toIndexes() ==> [5,6,7,8,9,10]
     * </pre>
     *
     * @return An array of integers
     */
    public int[] toIndexes() {
        if (this.nil) {
            return new int[0];
        } else {
            int count = this.to - this.from + 1;
            int[] indexes = new int[count];
            for (int i = 0; i < indexes.length; i++) {
                indexes[i] = this.from + i;
            }
            return indexes;
        }
    }

    /**
     * Returns a representation string for this range.
     * <p/>
     * Examples:
     * <pre>
     * Range.nil().toString() ==> "[]"
     * new Range(1).toString() ==> "[1]"
     * new Range(1,2).toString() ==> "[1..2]"
     * new Range(5,10).toString() ==> "[5..10]"
     * </pre>
     *
     * @return A representation
     */
    @Override
    public String toString() {
        if (this.nil) {
            return "[]";
        } else if (this.from == this.to) {
            return "[" + this.from + "]";
        } else {
            return "[" + this.from + ".." + this.to + "]";
        }
    }

    public static <T, K> IndexedArray<T, K> toIndexedArray(int from, int to, Function<Integer, T> creator, Function<T, K> indexer) {
        // Creates the range
        Array<Integer> range = Range.toArray(from, to);
        // Transforms into items
        Array<T> items = range.transform(creator);
        // Indexes
        return new IndexedArray<T, K>(indexer, items);
    }

    public static <T> List<T> toList(int from, int to, Function<Integer, T> creator) {
        // Creates the range
        Array<Integer> range = Range.toArray(from, to);
        // Transforms into items
        Array<T> items = range.transform(creator);
        // Converts
        return items.toList();
    }
}
