package net.sf.sido.spring.mvc;

import java.io.IOException;
import java.util.Map;

import javax.servlet.ServletInputStream;
import javax.servlet.ServletRequest;

import net.sf.sido.DOTypeNotFoundException;
import net.sf.sido.DataFactory;
import net.sf.sido.DataObject;
import net.sf.sido.DataType;
import net.sf.sido.Property;
import net.sf.sido.ValidationResults;
import net.sf.sido.io.DataObjectReader;
import net.sf.sido.io.json.JSONDataObjectReader;
import net.sf.sido.io.xml.XMLDataObjectReader;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.MethodParameter;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.support.WebArgumentResolver;
import org.springframework.web.context.request.NativeWebRequest;

@Component
public class DataObjectArgumentResolver implements WebArgumentResolver {

	public static final String REQUEST_PARAMETER_TYPE = "sido:type";

	private final DataFactory dataFactory;

	@Autowired
	public DataObjectArgumentResolver(DataFactory dataFactory) {
		this.dataFactory = dataFactory;
	}

	protected void check(DataObject o, MethodParameter methodParameter) {
		HasDataType hasDataType = methodParameter.getParameterAnnotation(HasDataType.class);
		if (hasDataType != null && hasDataType.validate()) {
			ValidationResults results = o.validate();
			if (!results.isValid()) {
				throw new IllegalStateException(String.format("Cannot validate DataObject %s: %s", methodParameter.getParameterName(), results.getErrors()));
			}
		}
	}

	protected DataType getType(MethodParameter methodParameter, NativeWebRequest webRequest, boolean typeRequired) {
		// Gets the DataType annotation
		HasDataType hasDataType = methodParameter.getParameterAnnotation(HasDataType.class);
		if (hasDataType != null) {
			String dataTypeName = hasDataType.value();
			return getTypeFromName(dataTypeName);
		} else {
			String typeName = webRequest.getParameter(REQUEST_PARAMETER_TYPE);
			if (StringUtils.isBlank(typeName)) {
				if (typeRequired) {
					throw new IllegalStateException(String.format("Type for DataObject %s is required. You can use a HasDataType annotation on the parameter or add a %s parameter in the request.",
							methodParameter.getParameterName(), REQUEST_PARAMETER_TYPE));
				} else {
					return null;
				}
			} else {
				return getTypeFromName(typeName);
			}
		}
	}

	protected DataType getTypeFromName(String dataTypeName) {
		DataType dataType = dataFactory.getType(dataTypeName);
		if (dataType == null) {
			throw new DOTypeNotFoundException(dataTypeName);
		} else {
			return dataType;
		}
	}

	@Override
	public Object resolveArgument(MethodParameter methodParameter, NativeWebRequest webRequest) throws Exception {
		if (DataObject.class.equals(methodParameter.getParameterType())) {
			return resolveInternal(methodParameter, webRequest);
		} else {
			return UNRESOLVED;
		}
	}

	protected <T> DataObject resolveArgumentFromInputStream(MethodParameter methodParameter, NativeWebRequest webRequest, DataObjectReader<T> reader) throws IOException {
		ServletRequest req = (ServletRequest) webRequest.getNativeRequest();
		ServletInputStream in = req.getInputStream();
		try {
			// Resolves the expected type
			DataType dataType = getType(methodParameter, webRequest, false);
			// Reads the object
			DataObject o = reader.read(in, dataFactory, dataType);
			// Checks the created object
			check(o, methodParameter);
			return o;
		} finally {
			in.close();
		}
	}

	protected DataObject resolveArgumentFromParameters(MethodParameter methodParameter, NativeWebRequest webRequest) {
		// Resolves the expected type
		DataType dataType = getType(methodParameter, webRequest, true);
		// Parameters
		HasDataType hasDataType = methodParameter.getParameterAnnotation(HasDataType.class);
		// Creates the instance
		DataObject o = dataType.newInstance();
		// Initialises the type?
		if (hasDataType != null && hasDataType.init()) {
			o.init(true);
		}
		// For all parameters
		Map<String, String[]> parameters = webRequest.getParameterMap();
		for(Map.Entry<String, String[]> entry : parameters.entrySet()) {
			String name = entry.getKey();
			String[] values = entry.getValue();
			Property<Object> property = dataType.getProperty(name);
			if (property != null) {
				// Sets the value from a string
				o.setAsString(name, values);
			}
		}
		// Checks the created object
		check(o, methodParameter);
		// OK
		return o;
	}

	protected DataObject resolveInternal(MethodParameter methodParameter, NativeWebRequest webRequest) throws IOException {
		ServletRequest req = webRequest.getNativeRequest(ServletRequest.class);
		// JSON content
		String contentType = req.getContentType();
		if (contentType != null && contentType.startsWith("sido/json")) {
			return resolveArgumentFromInputStream(methodParameter, webRequest, new JSONDataObjectReader());
		}
		// XML content
		else if (contentType != null && contentType.startsWith("sido/xml")) {
			return resolveArgumentFromInputStream(methodParameter, webRequest, new XMLDataObjectReader());
		}
		// Parameters
		else {
			return resolveArgumentFromParameters(methodParameter, webRequest);
		}
	}

}
