/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.snowflake.client.annotations.RunOnLinuxOrMac;
import net.snowflake.client.core.FileCacheManager;
import net.snowflake.client.core.FileUtil;
import net.snowflake.client.core.StmtUtil;
import net.snowflake.client.jdbc.BaseJDBCTest;
import net.snowflake.client.jdbc.SnowflakeUtil;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Tag;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.mockito.ArgumentMatchers;
import org.mockito.MockedStatic;
import org.mockito.Mockito;
import org.mockito.stubbing.Answer;

@Nested
@Tag(value="core")
class FileCacheManagerTest
extends BaseJDBCTest {
    private static final String CACHE_FILE_NAME = "credential_cache_v1.json.json";
    private static final String CACHE_DIR_PROP = "net.snowflake.jdbc.temporaryCredentialCacheDir";
    private static final String CACHE_DIR_ENV = "SF_TEMPORARY_CREDENTIAL_CACHE_DIR";
    private static final long CACHE_FILE_LOCK_EXPIRATION_IN_SECONDS = 60L;
    private FileCacheManager fileCacheManager;
    private File cacheFile;

    FileCacheManagerTest() {
    }

    @BeforeEach
    public void setup() throws IOException {
        this.fileCacheManager = FileCacheManager.builder().setCacheDirectorySystemProperty(CACHE_DIR_PROP).setCacheDirectoryEnvironmentVariable(CACHE_DIR_ENV).setBaseCacheFileName(CACHE_FILE_NAME).setCacheFileLockExpirationInSeconds(60L).build();
        this.cacheFile = this.createCacheFile();
    }

    @AfterEach
    public void clean() throws IOException {
        if (Files.exists(this.cacheFile.toPath(), new LinkOption[0])) {
            Files.delete(this.cacheFile.toPath());
        }
        if (Files.exists(this.cacheFile.getParentFile().toPath(), new LinkOption[0])) {
            Files.delete(this.cacheFile.getParentFile().toPath());
        }
    }

    @ParameterizedTest
    @CsvSource(value={"rwx------,rwx------,false", "rw-------,rwx------,true", "rw-------,rwx--xrwx,true", "r-x------,rwx------,false", "r--------,rwx------,true", "rwxrwx---,rwx------,false", "rwxrw----,rwx------,false", "rwxr-x---,rwx------,false", "rwxr-----,rwx------,false", "rwx-wx---,rwx------,false", "rwx-w----,rwx------,false", "rwx--x---,rwx------,false", "rwx---rwx,rwx------,false", "rwx---rw-,rwx------,false", "rwx---r-x,rwx------,false", "rwx---r--,rwx------,false", "rwx----wx,rwx------,false", "rwx----w-,rwx------,false", "rwx-----x,rwx------,false"})
    @RunOnLinuxOrMac
    public void throwWhenReadCacheFileWithPermissionDifferentThanReadWriteForUserTest(String permission, String parentDirectoryPermissions, boolean isSucceed) throws IOException {
        this.fileCacheManager.overrideCacheFile(this.cacheFile);
        Files.setPosixFilePermissions(this.cacheFile.toPath(), PosixFilePermissions.fromString(permission));
        Files.setPosixFilePermissions(this.cacheFile.getParentFile().toPath(), PosixFilePermissions.fromString(parentDirectoryPermissions));
        if (isSucceed) {
            Assertions.assertDoesNotThrow(() -> this.fileCacheManager.readCacheFile());
        } else {
            SecurityException ex = (SecurityException)Assertions.assertThrows(SecurityException.class, () -> this.fileCacheManager.readCacheFile());
            Assertions.assertTrue((boolean)ex.getMessage().contains("is wider than allowed."));
        }
    }

    @Test
    @RunOnLinuxOrMac
    public void notThrowExceptionWhenCacheFolderIsNotAccessible() throws IOException {
        try {
            Files.setPosixFilePermissions(this.cacheFile.getParentFile().toPath(), PosixFilePermissions.fromString("---------"));
            FileCacheManager fcm = FileCacheManager.builder().setCacheDirectorySystemProperty(CACHE_DIR_PROP).setCacheDirectoryEnvironmentVariable(CACHE_DIR_ENV).setBaseCacheFileName(CACHE_FILE_NAME).setCacheFileLockExpirationInSeconds(60L).build();
            Assertions.assertDoesNotThrow(() -> ((FileCacheManager)fcm).readCacheFile());
        }
        finally {
            Files.setPosixFilePermissions(this.cacheFile.getParentFile().toPath(), PosixFilePermissions.fromString("rwx------"));
        }
    }

    @Test
    @RunOnLinuxOrMac
    public void throwWhenOverrideCacheFileHasDifferentOwnerThanCurrentUserTest() {
        try (MockedStatic fileUtilMock = Mockito.mockStatic(FileUtil.class, (Answer)Mockito.CALLS_REAL_METHODS);){
            fileUtilMock.when(() -> FileUtil.getFileOwnerName((Path)((Path)ArgumentMatchers.isA(Path.class)))).thenReturn((Object)"anotherUser");
            SecurityException ex = (SecurityException)Assertions.assertThrows(SecurityException.class, () -> this.fileCacheManager.readCacheFile());
            Assertions.assertTrue((boolean)ex.getMessage().contains("The file owner is different than current user"));
        }
    }

    @Test
    @RunOnLinuxOrMac
    public void notThrowForToWidePermissionsWhenOnlyOwnerPermissionsSetFalseTest() throws IOException {
        this.fileCacheManager.setOnlyOwnerPermissions(false);
        Files.setPosixFilePermissions(this.cacheFile.toPath(), PosixFilePermissions.fromString("rwxrwx---"));
        Assertions.assertDoesNotThrow(() -> this.fileCacheManager.readCacheFile());
    }

    @Test
    @RunOnLinuxOrMac
    public void throwWhenOverrideCacheFileNotFound() {
        Path wrongPath = Paths.get(SnowflakeUtil.systemGetProperty((String)"user.home"), ".cache", "snowflake2", "wrongFileName");
        SecurityException ex = (SecurityException)Assertions.assertThrows(SecurityException.class, () -> this.fileCacheManager.overrideCacheFile(wrongPath.toFile()));
        Assertions.assertTrue((boolean)ex.getMessage().contains("Unable to access the file/directory to check the permissions. Error: java.nio.file.NoSuchFileException:"));
    }

    @Test
    @RunOnLinuxOrMac
    public void throwWhenSymlinkAsCache() throws IOException {
        Path symlink = this.createSymlink();
        try {
            SecurityException ex = (SecurityException)Assertions.assertThrows(SecurityException.class, () -> this.fileCacheManager.overrideCacheFile(symlink.toFile()));
            Assertions.assertTrue((boolean)ex.getMessage().contains("Symbolic link is not allowed for file cache"));
        }
        finally {
            if (Files.exists(symlink, new LinkOption[0])) {
                Files.delete(symlink);
            }
        }
    }

    private File createCacheFile() {
        Path cacheFile = Paths.get(SnowflakeUtil.systemGetProperty((String)"user.home"), ".cache", "snowflake_cache", CACHE_FILE_NAME);
        try {
            if (Files.exists(cacheFile, new LinkOption[0])) {
                Files.delete(cacheFile);
            }
            if (Files.exists(cacheFile.getParent(), new LinkOption[0])) {
                Files.delete(cacheFile.getParent());
            }
            if (!SnowflakeUtil.isWindows()) {
                Files.createDirectories(cacheFile.getParent(), PosixFilePermissions.asFileAttribute(Stream.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE).collect(Collectors.toSet())));
            } else {
                Files.createDirectories(cacheFile.getParent(), new FileAttribute[0]);
            }
            if (!SnowflakeUtil.isWindows()) {
                Files.createFile(cacheFile, PosixFilePermissions.asFileAttribute(Stream.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE).collect(Collectors.toSet())));
            } else {
                Files.createFile(cacheFile, new FileAttribute[0]);
            }
            ObjectNode cacheContent = StmtUtil.mapper.createObjectNode();
            cacheContent.put("token", "tokenValue");
            this.fileCacheManager.overrideCacheFile(cacheFile.toFile());
            this.fileCacheManager.writeCacheFile((JsonNode)cacheContent);
            return cacheFile.toFile();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private Path createSymlink() throws IOException {
        Path link = Paths.get(this.cacheFile.getParent(), "symlink_credential_cache_v1.json.json");
        if (Files.exists(link, new LinkOption[0])) {
            Files.delete(link);
        }
        return Files.createSymbolicLink(link, this.cacheFile.toPath(), new FileAttribute[0]);
    }

    @Test
    void shouldCreateDirAndFile() {
        String tmpDirPath = System.getProperty("java.io.tmpdir");
        String cacheDirPath = tmpDirPath + File.separator + "snowflake-cache-dir";
        System.setProperty("FILE_CACHE_MANAGER_SHOULD_CREATE_DIR_AND_FILE", cacheDirPath);
        FileCacheManager.builder().setOnlyOwnerPermissions(false).setCacheDirectorySystemProperty("FILE_CACHE_MANAGER_SHOULD_CREATE_DIR_AND_FILE").setBaseCacheFileName("cache-file").build();
        Assertions.assertTrue((boolean)new File(tmpDirPath).exists());
    }
}

