/*
 * This file is part of JBizMo, a set of tools, libraries and plug-ins
 * for modeling and creating Java-based enterprise applications.
 * For more information visit:
 *
 * http://sourceforge.net/projects/jbizmo/
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package net.sourceforge.jbizmo.commons.selenium.driver.imp;

import java.lang.invoke.MethodHandles;
import java.lang.reflect.Constructor;
import java.util.concurrent.TimeUnit;

import net.sourceforge.jbizmo.commons.selenium.driver.IDriverProducer;
import net.sourceforge.jbizmo.commons.selenium.util.SeleniumTestProperties;

import org.openqa.selenium.WebDriver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * Standard implementation of a producer for Selenium WebDriver objects
 * </p>
 * <p>
 * Copyright 2017 (C) by Martin Ganserer
 * </p>
 * @author Martin Ganserer
 * @version 1.0.0
 */
public class StandardDriverProducer implements IDriverProducer
{
	private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());

	private SeleniumTestProperties properties;

	/*
	 * (non-Javadoc)
	 * @see net.sourceforge.jbizmo.commons.selenium.driver.IDriverProducer#setProperties(net.sourceforge.jbizmo.commons.selenium.util.SeleniumTestProperties)
	 */
	@Override
	public void setProperties(SeleniumTestProperties properties)
	{
		this.properties = properties;
	}

	/*
	 * (non-Javadoc)
	 * @see net.sourceforge.jbizmo.commons.selenium.driver.IDriverProducer#getDriverInstance()
	 */
	@Override
	public WebDriver getDriverInstance()
	{
		final String driverName = properties.getDriverName();
		final String driverPath = properties.getDriverPath();

		if(driverName != null && !driverName.isEmpty() && driverPath != null && !driverPath.isEmpty())
			System.setProperty(driverName, driverPath);

		try
		{
			logger.debug("Create driver '{}'", properties.getDriverClassName());

			final Class<? extends WebDriver> webDriverClass = Class.forName(properties.getDriverClassName()).asSubclass(WebDriver.class);
			final Constructor<? extends WebDriver> constructor = webDriverClass.getConstructor();

			final WebDriver driver = constructor.newInstance();
			driver.manage().timeouts().implicitlyWait(properties.getImplicitWaitTime(), TimeUnit.SECONDS);
			driver.manage().timeouts().pageLoadTimeout(properties.getPageLoadTimeout(), TimeUnit.SECONDS);

			if(properties.isMaximizeWindow())
				driver.manage().window().maximize();

			return driver;
		}
		catch (final Exception e)
		{
			final String message = "Error while creating Selenium WebDriver!";

			logger.error(message, e);

			throw new RuntimeException(e);
		}
	}

}
