/*
 * This file is part of JBizMo, a set of tools, libraries and plug-ins
 * for modeling and creating Java-based enterprise applications.
 * For more information visit:
 *
 * http://sourceforge.net/projects/jbizmo/
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package net.sourceforge.jbizmo.commons.selenium.junit;

import net.sourceforge.jbizmo.commons.selenium.data.ITestDataProvider;
import net.sourceforge.jbizmo.commons.selenium.util.SeleniumTestProperties;

import org.openqa.selenium.WebDriver;

/**
 * <p>
 * The test context is responsible for providing essential objects for Selenium tests
 * </p>
 * <p>
 * Copyright 2017 (C) by Martin Ganserer
 * </p>
 * @author Martin Ganserer
 * @version 1.0.0
 */
public class SeleniumTestContext
{
	private final WebDriver driver;
	private final ITestDataProvider dataProvider;
	private final SeleniumTestProperties properties;
	private boolean closeDriver;
	private String baseURL;
	private boolean securityContextAdded;

	/**
	 * Constructor
	 * @param driver
	 * @param dataProvider
	 * @param properties
	 */
	public SeleniumTestContext(WebDriver driver, ITestDataProvider dataProvider, SeleniumTestProperties properties)
	{
		this.driver = driver;
		this.dataProvider = dataProvider;
		this.properties = properties;
		this.baseURL = properties.getBaseURL();
	}

	/**
	 * @return the Selenium WebDriver
	 */
	public WebDriver getDriver()
	{
		return driver;
	}

	/**
	 * @return the test data provider
	 */
	public ITestDataProvider getDataProvider()
	{
		return dataProvider;
	}

	/**
	 * @return the base URL
	 */
	public String getBaseURL()
	{
		return baseURL;
	}

	/**
	 * @return true if the driver should be closed after finishing the test
	 */
	public boolean isCloseDriver()
	{
		return closeDriver;
	}

	/**
	 * @param closeDriver
	 */
	public void setCloseDriver(boolean closeDriver)
	{
		this.closeDriver = closeDriver;
	}

	/**
	 * @return the global test properties
	 */
	public SeleniumTestProperties getProperties()
	{
		return properties;
	}

	/**
	 * Append the base URL with the name of the context that contains protected resources.
	 * @param securityContextName
	 */
	public void addSecurityContextToBaseURL(String securityContextName)
	{
		// Avoid creating an invalid base URL if the method is called multiple times within a test!
		if(securityContextAdded)
			return;

		if(securityContextName == null || securityContextName.isEmpty())
			return;

		if(!baseURL.endsWith("/") && !securityContextName.startsWith("/"))
			baseURL += "/";

		baseURL += securityContextName;
		securityContextAdded = true;
	}

	/**
	 * Remove the name of the security context from the base URL
	 * @param securityContextName
	 */
	public void removeSecurityContextFromBaseURL(String securityContextName)
	{
		// Don't try to remove the context name from the base URL if it hasn't been added before!
		if(!securityContextAdded)
			return;

		if(securityContextName == null || securityContextName.isEmpty())
			return;

		final int pos = baseURL.lastIndexOf(securityContextName);

		if(pos > 0)
			baseURL = baseURL.substring(0, pos);

		securityContextAdded = false;
	}

	/**
	 * Causes the currently running thread to sleep
	 * @param millis the length of time to sleep in milliseconds
	 * @throws IllegalArgumentException if the value of {@code millis} is negative
	 */
	public void delayTest(int millis)
	{
		try
		{
			Thread.sleep(millis);
		}
		catch (final InterruptedException e)
		{
			// Ignored!
		}
	}

}
