/*
 * This file is part of JBizMo, a set of tools, libraries and plug-ins
 * for modeling and creating Java-based enterprise applications.
 * For more information visit:
 *
 * http://sourceforge.net/projects/jbizmo/
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package net.sourceforge.jbizmo.commons.selenium.page.imp.primefaces;

import java.lang.reflect.Constructor;
import java.util.concurrent.TimeUnit;

import net.sourceforge.jbizmo.commons.selenium.junit.SeleniumTestContext;
import net.sourceforge.jbizmo.commons.selenium.page.AbstractPageComponent;

import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.support.ui.FluentWait;

import com.google.common.base.Predicate;

/**
 * <p>
 * Base class for all page object components of a Primefaces application
 * </p>
 * <p>
 * Copyright 2017 (C) by Martin Ganserer
 * </p>
 * @author Martin Ganserer
 * @version 1.0.0
 */
public abstract class AbstractPrimefacesPageComponent extends AbstractPageComponent
{
	private static final String JS_AJAX_QUEUE_EMPTY = "if(window.PrimeFaces){return PrimeFaces.ajax.Queue.isEmpty();}else{return true;}";

	final long ajaxTimeout;

	/**
	 * Constructor
	 * @param testContext
	 */
	public AbstractPrimefacesPageComponent(SeleniumTestContext testContext)
	{
		super(testContext);

		this.ajaxTimeout = testContext.getProperties().getAjaxTimeout();
	}

	/*
	 * (non-Javadoc)
	 * @see net.sourceforge.jbizmo.commons.selenium.page.AbstractPageComponent#waitForPendingAjaxRequests()
	 */
	@Override
	public void waitForPendingAjaxRequests()
	{
		if(ajaxTimeout <= 0)
		{
			logger.trace("Skip waiting for pending AJAX requests");

			return;
		}

		if(!(driver instanceof JavascriptExecutor))
		{
			logger.warn("Driver doesn't support execution of JavaScript!");

			return;
		}

		logger.trace("Wait for pending AJAX requests");

		final FluentWait<WebDriver> wait = new FluentWait<>(driver);
		wait.withTimeout(ajaxTimeout, TimeUnit.SECONDS);
		wait.pollingEvery(AJAX_POLLING_INTERVAL_MILLISECONDS, TimeUnit.MILLISECONDS);

		wait.until((Predicate<WebDriver>) driver ->
		{
			final JavascriptExecutor js = (JavascriptExecutor) driver;
			boolean finished;

			try
			{
				finished = Boolean.TRUE.equals(js.executeScript(JS_AJAX_QUEUE_EMPTY));

				if(finished)
					logger.trace("No pending AJAX requests found!");
				else
					logger.trace("AJAX event queue is not empty...");
			}
			catch (final Exception e)
			{
				logger.error("Error while executing JavaScript!", e);

				// If the JavaScript invocation fails the wait loop should be finished!
				return true;
			}

			return finished;
		});
	}

	/**
	 * Create an instance of the selected page object class
	 * @param <T> the type of the page object to be returned
	 * @param pageClass
	 * @return an instance of the selected page class
	 * @throws AssertionError if the creation of the page object has failed
	 */
	protected <T extends AbstractPageObject> T createPageObject(Class<T> pageClass)
	{
		delayPageLoad();

		try
		{
			final Constructor<T> constructor = pageClass.getConstructor(SeleniumTestContext.class);

			return constructor.newInstance(testContext);
		}
		catch (final Exception e)
		{
			fail("Could not create page object '" + pageClass.getName() + "'!", e);

			return null;
		}
	}

}
