/*
 * This file is part of JBizMo, a set of tools, libraries and plug-ins
 * for modeling and creating Java-based enterprise applications.
 * For more information visit:
 *
 * http://sourceforge.net/projects/jbizmo/
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package net.sourceforge.jbizmo.commons.selenium.page.imp.vaadin;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.List;

import net.sourceforge.jbizmo.commons.selenium.data.PageActionResult;
import net.sourceforge.jbizmo.commons.selenium.data.PageElementTestData;
import net.sourceforge.jbizmo.commons.selenium.junit.SeleniumTestContext;

import org.openqa.selenium.Keys;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.interactions.Actions;

/**
 * <p>
 * Abstract base class for all page objects of a Vaadin application
 * </p>
 * <p>
 * Copyright 2017 (C) by Martin Ganserer
 * </p>
 * @author Martin Ganserer
 * @version 1.0.0
 */
public abstract class AbstractPageObject extends AbstractVaadinPageComponent
{
	public static final String BUTTON_ID_LOG_IN = "cmdLogin";

	private static final String SLASH = "/";
	private static final String ITEM_CONTAINER_XPATH = "//div[@id='VAADIN_COMBOBOX_OPTIONLIST']/div/div/table/tbody/tr/td/span";
	private static final String COMBO_BOX_STATUS_XPATH = "//div[@id='VAADIN_COMBOBOX_OPTIONLIST']/div/div[@class='v-filterselect-status']";
	private static final String NAVIGATOR_XPATH = "//table[@role='treegrid']/tbody/tr/td/div/";
	private static final String NAVIGATOR_FOLDER_XPATH = NAVIGATOR_XPATH + "span[contains(@class,'v-tree8-expander collapsed')]";
	private static final String ATTR_NAME_VALUE = "value";
	private static final String BUTTON_ID_SAVE = "cmdSave";
	private static final String BUTTON_ID_CANCEL = "cmdCancel";
	private static final String BUTTON_ID_LOG_OUT = "cmdLogout";
	private static final String LABEL_ID_PAGE_TITLE = "lblPageTitle";
	private static final int COMBO_BOX_SCROLL_KEY_COUNT = 5;
	private static final int ARROW_ICON_OFFSET_X = 5;
	private static final int ARROW_ICON_OFFSET_Y = 5;

	/**
	 * Constructor
	 * @param testContext
	 */
	public AbstractPageObject(SeleniumTestContext testContext)
	{
		super(testContext);
	}

	/**
	 * Open page
	 * @param resourcePath
	 * @throws AssertionError if the parameter <code><resourcePath</code> is null
	 */
	public void open(String resourcePath)
	{
		open(resourcePath, null);
	}

	/**
	 * Open page to display data for a specific object identified by the given ID
	 * @param resourcePath
	 * @param objectId
	 * @throws AssertionError if the parameter <code><resourcePath</code> is null
	 */
	public void open(String resourcePath, String objectId)
	{
		assertNotNull("Parameter 'resourcePath' must not be null!", resourcePath);

		waitForPendingAjaxRequests();

		try
		{
			String url = buildPageURL(testContext.getBaseURL(), resourcePath);

			if(objectId != null && !objectId.isEmpty())
				url += SLASH + URLEncoder.encode(objectId, StandardCharsets.UTF_8.name());

			logger.debug("Open URL '{}'", url);

			driver.get(url);
		}
		catch (final UnsupportedEncodingException e)
		{
			logger.error("Encoding not supported!", e);
		}
	}

	/**
	 * Navigate to a page by selecting a tree view item with the specified navigation target! As the page object is created via introspection it is necessary that the respective class provides an
	 * appropriate constructor!
	 * @param <T> the type of the page object that should be returned
	 * @param navigationTarget
	 * @param pageClass
	 * @return a page instance whose type is defined by the respective parameter
	 * @throws AssertionError if the page object could not be instantiated
	 */
	public <T extends AbstractPageObject> T openPageByNavigator(String navigationTarget, Class<T> pageClass)
	{
		logger.debug("Navigate to '{}'", navigationTarget);

		// Iterate over all tree items as long as all folder items are expanded!
		while(true)
		{
			final List<WebElement> folderItems = findWebElementsByXPath(NAVIGATOR_FOLDER_XPATH);
			boolean itemFound = false;

			for(final WebElement item : folderItems)
			{
				logger.trace("Expand navigator tree item '{}'", item.getText());

				// We must click the arrow icon in order to expand the tree item!
				new Actions(driver).moveToElement(item, ARROW_ICON_OFFSET_X, ARROW_ICON_OFFSET_Y).click().build().perform();

				itemFound = true;
			}

			// Exit the loop if no collapsed tree item could be found!
			if(!itemFound)
				break;
		}

		// We assume that the label uniquely identifies the view to be opened!
		final String expression = NAVIGATOR_XPATH + "div/span[text()='" + navigationTarget + "']";

		// Search for the appropriate tree item
		WebElement treeItem = findWebElementByXPath(expression);

		// Click on it in order to select it
		new Actions(driver).click(treeItem).build().perform();

		// Search for the tree item again in order to avoid a StaleElementReferenceException!
		treeItem = findWebElementByXPath(expression);

		// Click on the tree item again in order to open the navigation target
		new Actions(driver).click(treeItem).build().perform();

		return createPageObject(pageClass);
	}

	/**
	 * Press logout button
	 * @throws AssertionError if the button could not be found
	 */
	public void pressLogoutButton()
	{
		pressButton(BUTTON_ID_LOG_OUT);
	}

	/**
	 * Press the 'Save' button
	 * @throws AssertionError if the button could not be found
	 */
	public void pressSaveButton()
	{
		pressButton(BUTTON_ID_SAVE);
	}

	/**
	 * Press the 'Save' button
	 * @param pageClass
	 * @param <T> the type of the page object that should be returned
	 * @return a page instance whose type is defined by the respective parameter
	 * @throws AssertionError if the button either could not be found, or the page object could not be created
	 */
	public <T extends AbstractPageObject> T pressSaveButton(Class<T> pageClass)
	{
		pressSaveButton();

		return createPageObject(pageClass);
	}

	/**
	 * Press the 'Cancel' button
	 * @throws AssertionError if the button could not be found
	 */
	public void pressCancelButton()
	{
		pressButton(BUTTON_ID_CANCEL);
	}

	/**
	 * Press a button with a given ID
	 * @param id
	 * @throws AssertionError if the button could not be found
	 */
	public void pressButton(String id)
	{
		logger.debug("Press button with ID '{}'", id);

		findWebElement(id).click();
	}

	/**
	 * Change the selection of a checkbox field
	 * @param testData the field's test data object that provides necessary information
	 * @throws AssertionError if changing of the checkbox selection has failed
	 */
	public void setCheckBoxValue(PageElementTestData testData)
	{
		assertNotNull("Value for checkbox field '" + testData.getElementId() + "' must not be null!", testData.getNewValue());

		logger.debug("Set selection of checkbox '{}'", testData.getElementId());

		final WebElement inputField = findWebElementByXPath("//span[@id='" + testData.getElementId() + "']/input");
		final boolean newSelection = testData.getNewValue().equalsIgnoreCase(Boolean.toString(true));
		final boolean currentSelection = inputField.isSelected();

		if(currentSelection)
			logger.debug("Checkbox '{}' is selected", testData.getElementId());
		else
			logger.debug("Checkbox '{}' is not selected", testData.getElementId());

		// Skip further operations if the current value and the new value are equal!
		if(currentSelection == newSelection)
			return;

		logger.debug("Change selection of checkbox '{}'", testData.getElementId());

		// Send a space character to the field in order to toggle its selection
		inputField.sendKeys(Keys.SPACE);
	}

	/**
	 * Validate selection of checkbox field
	 * @param testData the field's test data object that provides necessary information
	 * @throws AssertionError if the validation has failed
	 */
	public void validateCheckBoxValue(PageElementTestData testData)
	{
		assertNotNull("Expected value for checkbox field '" + testData.getElementId() + "' must not be null!", testData.getExpectedValue());

		logger.debug("Validate selection of checkbox '{}'", testData.getElementId());

		final boolean expectedSelection = testData.getExpectedValue().equalsIgnoreCase(Boolean.toString(true));
		final boolean currentSelection = findWebElementByXPath("//span[@id='" + testData.getElementId() + "']/input").isSelected();
		final String message = "Validation of checkbox '" + testData.getElementId() + "' selection has failed!";

		assertTrue(message, expectedSelection == currentSelection);
	}

	/**
	 * Select an item of an auto-complete field
	 * @param testData the field's test data object that provides necessary information
	 * @throws AssertionError if an item could not be selected
	 */
	public void selectAutoCompleteItem(PageElementTestData testData)
	{
		assertNotNull("Value for auto-complete field '" + testData.getElementId() + "' must not be null!", testData.getNewValue());

		logger.debug("Enter text '{}' into auto-complete field '{}'", testData.getNewValue(), testData.getElementId());

		// Clear the input field and enter search condition
		final WebElement inputField = findWebElementByXPath("//div[@id='" + testData.getElementId() + "']/input");
		inputField.clear();

		final String inputText = testData.getNewValue();

		inputField.sendKeys(inputText);

		if(inputText.length() > 1)
		{
			// Remove last input character in order to force the component to display the available items!
			inputField.sendKeys(Keys.BACK_SPACE);

			// Add last character again
			inputField.sendKeys(inputText.substring(inputText.length() - 1));
		}

		final boolean itemFound = searchAndSelectItem(testData.getNewValue());

		if(!itemFound)
			fail("Could not find selectable item '" + testData.getNewValue() + "' for auto-complete field '" + testData.getElementId() + "'!");
	}

	/**
	 * Validate the selected item of an auto-complete field
	 * @param testData the field's test data object that provides necessary information
	 * @throws AssertionError if the validation has failed
	 */
	public void validateAutoCompleteItem(PageElementTestData testData)
	{
		assertNotNull("Expected value for auto-complete field '" + testData.getElementId() + "' must not be null!", testData.getExpectedValue());

		final String fieldId = testData.getElementId();

		logger.debug("Validate the selected item of auto-complete field '{}'", fieldId);

		final WebElement inputField = findWebElementByXPath("//div[@id='" + fieldId + "']/input");
		final String message = "Validation of auto-complete field '" + fieldId + "' has failed!";

		assertEquals(message, testData.getExpectedValue(), inputField.getAttribute(ATTR_NAME_VALUE));
	}

	/**
	 * Select an item of a combobox field
	 * @param testData the field's test data object that provides necessary information
	 * @throws AssertionError if an item could not be selected
	 */
	public void selectComboboxItem(PageElementTestData testData)
	{
		assertNotNull("Value for combobox field '" + testData.getElementId() + "' must not be null!", testData.getNewValue());

		logger.debug("Search for item '{}' in combobox '{}'", testData.getNewValue(), testData.getElementId());

		// Search for the combobox field and click on it in order to open a pop-up dialog containing the items
		final WebElement combobox = findWebElementByXPath("//div[@id='" + testData.getElementId() + "']/input");
		combobox.click();

		boolean itemFound = searchItem(combobox, testData.getNewValue(), true);

		if(!itemFound)
			itemFound = searchItem(combobox, testData.getNewValue(), false);

		if(!itemFound)
			fail("Could not find selectable item '" + testData.getNewValue() + "' for combobox '" + testData.getElementId() + "'!");
	}

	/**
	 * Validate the selected item of a combobox field
	 * @param testData the field's test data object that provides necessary information
	 * @throws AssertionError if the validation has failed
	 */
	public void validateComboboxItem(PageElementTestData testData)
	{
		assertNotNull("Expected value for combobox field '" + testData.getElementId() + "' must not be null!", testData.getExpectedValue());

		final String fieldId = testData.getElementId();

		logger.debug("Validate the selected item of combobox '{}'", fieldId);

		final WebElement combobox = findWebElementByXPath("//div[@id='" + testData.getElementId() + "']/input");
		final String message = "Validation of combobox '" + fieldId + "' has failed!";

		assertEquals(message, testData.getExpectedValue(), combobox.getAttribute(ATTR_NAME_VALUE));
	}

	/**
	 * Select an item by opening a 'List of Values' in a pop-up window
	 * @param testData the field's test data object that provides necessary information
	 */
	public void selectLoVItem(PageElementTestData testData)
	{
		assertNotNull("Value for LoV field '" + testData.getElementId() + "' must not be null!", testData.getNewValue());

		logger.debug("Open LoV dialog");

		// Open LoV by pressing respective button
		findWebElementByXPath("//div[@id='" + testData.getElementId() + "']//div[contains(@class, 'v-button')]").click();

		logger.debug("Enter text '{}' into LoV field '{}'", testData.getNewValue(), testData.getElementId());

		// Enter filter
		final WebElement txtFilterInput = findWebElementByXPath("//div[@class='v-window-contents']//input[@id='txtInput']");
		txtFilterInput.clear();
		txtFilterInput.sendKeys(testData.getNewValue());

		final DataTableComponent tableLoV = new DataTableComponent(testContext);

		// The table component must contain only one row!
		if(tableLoV.getRowCount() != 1)
			fail("LoV table must contain exactly one row!");

		final WebElement row = tableLoV.getRowByRowIndex(1);

		// Double-click on the row in order to apply the selection
		tableLoV.doubleClickRow(row);

		logger.debug("Close LoV dialog");
	}

	/**
	 * Validate the selected item of a LoV field
	 * @param testData the field's test data object that provides necessary information
	 * @throws AssertionError if the validation has failed
	 */
	public void validateLoVItem(PageElementTestData testData)
	{
		assertNotNull("Expected value for LoV field '" + testData.getElementId() + "' must not be null!", testData.getExpectedValue());

		final String fieldId = testData.getElementId();

		logger.debug("Validate the selected item of LoV field '{}'", fieldId);

		final WebElement txtLoV = findWebElementByXPath("//div[@id='" + testData.getElementId() + "']//div/input");
		final String message = "Validation of LoV field '" + fieldId + "' has failed!";

		assertEquals(message, testData.getExpectedValue(), txtLoV.getAttribute(ATTR_NAME_VALUE));
	}

	/**
	 * Enter a value into a date field
	 * @param testData the field's test data object that provides necessary information
	 */
	public void setDateFieldValue(PageElementTestData testData)
	{
		assertNotNull("Text for date field '" + testData.getElementId() + "' must not be null!", testData.getNewValue());

		logger.debug("Enter text '{}' into date field '{}'", testData.getNewValue(), testData.getElementId());

		final WebElement dateField = findWebElementByXPath("//div[@id='" + testData.getElementId() + "']/input");
		dateField.clear();
		dateField.sendKeys(testData.getNewValue());
	}

	/**
	 * Validate a date field by using the field's test data object
	 * @param testData the field's test data object that provides necessary information
	 * @throws AssertionError if the validation has failed
	 */
	public void validateDateFieldValue(PageElementTestData testData)
	{
		assertNotNull("Expected value for date field '" + testData.getElementId() + "' must not be null!", testData.getExpectedValue());

		logger.debug("Validate value of date field '{}'", testData.getElementId());

		final WebElement dateField = findWebElementByXPath("//div[@id='" + testData.getElementId() + "']/input");
		final String message = "Validation of date field '" + testData.getElementId() + "' has failed!";

		assertEquals(message, testData.getExpectedValue(), dateField.getAttribute(ATTR_NAME_VALUE));
	}

	/**
	 * Validate a label field by using the field's test data object
	 * @param testData the field's test data object that provides necessary information
	 * @throws AssertionError if the validation has failed
	 */
	public void validateLabelText(PageElementTestData testData)
	{
		assertNotNull("Expected value for label field '" + testData.getElementId() + "' must not be null!", testData.getExpectedValue());

		logger.debug("Validate text of label field '{}'", testData.getElementId());

		final WebElement labelField = findWebElement(testData.getElementId());
		final String message = "Validation of label field '" + testData.getElementId() + "' has failed!";

		assertEquals(message, testData.getExpectedValue(), labelField.getText());
	}

	/**
	 * Validate an internal link field by using the field's test data object
	 * @param testData the field's test data object that provides necessary information
	 * @throws AssertionError if the validation has failed
	 */
	public void validateInternalLinkText(PageElementTestData testData)
	{
		assertNotNull("Expected value for internal link field '" + testData.getElementId() + "' must not be null!", testData.getExpectedValue());

		logger.debug("Validate text of internal link field '{}'", testData.getElementId());

		final WebElement linkField = findWebElementByXPath("//div[@id='" + testData.getElementId() + "']//div/input");
		final String message = "Validation of internal link field '" + testData.getElementId() + "' has failed!";

		assertEquals(message, testData.getExpectedValue(), linkField.getAttribute(ATTR_NAME_VALUE));
	}

	/**
	 * Upload a file in a pop-up window
	 * @param testData the test data object that provides necessary information
	 * @throws AssertionError if the upload button either could not be found, or test data is invalid
	 */
	public void uploadFile(PageElementTestData testData)
	{
		assertNotNull("The file upload path must not be null!", testData.getNewValue());

		logger.debug("Upload file '{}'", testData.getNewValue());

		// Search for the 'Upload' button and click on it in order to open a pop-up dialog
		findWebElement(testData.getElementId()).click();

		// Search for the file input element and enter the path of the file to be uploaded
		findWebElementByXPath(PopUpDialog.POPUP_FILE_INPUT_XPATH).sendKeys(testData.getNewValue());
	}

	/**
	 * Open a tab page identified by the given ID
	 * @param tabPageId
	 * @throws AssertionError if the tab page could not be found
	 */
	public void openTabPage(String tabPageId)
	{
		logger.debug("Open tab page '{}'", tabPageId);

		findWebElement(tabPageId).click();
	}

	/**
	 * Wait for a message dialog and perform a status validation check
	 * @param actionResult
	 * @return a message dialog
	 * @throws AssertionError if the status validation either has failed, or an element could not be found
	 */
	public PopUpDialog waitForMessageDialog(PageActionResult actionResult)
	{
		logger.debug("Waiting for message dialog...");

		final PopUpDialog dlg = new PopUpDialog(this);
		dlg.validateStatus(actionResult);

		return dlg;
	}

	/**
	 * Compare the page title with the expected text provided by the test data object
	 * @param testData the test data object that provides necessary information
	 * @throws AssertionError if the validation either has failed, or the test data is invalid
	 */
	public void validatePageTitle(PageElementTestData testData)
	{
		assertNotNull("Expected page title must not be null!", testData.getExpectedValue());

		if(logger.isDebugEnabled())
			logger.debug("Validate page title '{}'", testData.getExpectedValue().trim());

		// As no reasonable page title is available an appropriate text provided by the page header area must be used!
		WebElement titleLabel = findWebElement(LABEL_ID_PAGE_TITLE);

		// It could be the case that the title is contained in another element
		final String refLabelId = titleLabel.getAttribute("aria-labelledby");

		// Search for this element if the respective reference target attribute exists
		if(refLabelId != null)
			titleLabel = findWebElementByXPath("//div[@id='" + refLabelId + "']/div");

		final String currentPageTitle = titleLabel.getText().trim();
		final String expectedTitle = testData.getExpectedValue().trim();
		final String message = "The current page title '" + currentPageTitle + "' and the expected text '" + expectedTitle + "' are different!";

		assertTrue(message, currentPageTitle.equals(expectedTitle));
	}

	/**
	 * Build page URL using the given base URL and a resource path. If necessary, both strings will be joined by using '/' character!
	 * @param baseURL
	 * @param resourcePath
	 * @return the URL
	 */
	protected String buildPageURL(String baseURL, String resourcePath)
	{
		if(resourcePath == null || resourcePath.isEmpty())
			return baseURL;

		if(!baseURL.endsWith(SLASH) && !resourcePath.startsWith(SLASH))
			return baseURL + SLASH + "#!" + resourcePath;

		return baseURL + "#!" + resourcePath;
	}

	/**
	 * Search for an item within a given combobox and scroll through the list until one end is reached
	 * @param comboBoxElement
	 * @param itemText
	 * @param scrollDown
	 * @return true if an item could be selected
	 */
	protected boolean searchItem(WebElement comboBoxElement, String itemText, boolean scrollDown)
	{
		// Depending on the scroll direction we must use a different key!
		final Keys arrowKey = scrollDown ? Keys.ARROW_DOWN : Keys.ARROW_UP;

		while(true)
		{
			if(searchAndSelectItem(itemText))
				return true;

			// The respective item wasn't found. Thus, we must scroll through the list!
			if(canScroll(scrollDown))
				break;

			if(scrollDown)
				logger.trace("Scroll down");
			else
				logger.trace("Scroll up");

			// By sending arrow keys it is possible to refresh the item list dynamically!
			for(int i = 0; i < COMBO_BOX_SCROLL_KEY_COUNT; i++)
				comboBoxElement.sendKeys(arrowKey);
		}

		return false;
	}

	/**
	 * Test if further scroll operations are possible by extracting dynamic page index information
	 * @param upperEnd
	 * @return true if the end of the item list is reached
	 */
	protected boolean canScroll(boolean upperEnd)
	{
		// Search for the element that contains the status
		final WebElement statusElement = findWebElementByXPath(COMBO_BOX_STATUS_XPATH);

		// Extract page index information
		final String[] indexArray = statusElement.getText().replace('/', '-').split("-");

		final int pageStartIndex = Integer.parseInt(indexArray[0].trim());
		final int pageEndIndex = Integer.parseInt(indexArray[1].trim());
		final int lastElementIndex = Integer.parseInt(indexArray[2].trim());

		if(upperEnd && pageEndIndex == lastElementIndex)
		{
			logger.trace("Can scroll up");

			return true;
		}

		if(!upperEnd && pageStartIndex == 1)
		{
			logger.trace("Can scroll down");

			return true;
		}

		logger.trace("Scrolling not possible");

		return false;
	}

	/**
	 * Search for a given item and select it if it is contained in the respective list
	 * @param item
	 * @return true if an item could be selected
	 */
	protected boolean searchAndSelectItem(String item)
	{
		final List<WebElement> itemElements = findWebElementsByXPath(ITEM_CONTAINER_XPATH);

		for(final WebElement itemElement : itemElements)
			if(item.equals(itemElement.getText()))
			{
				logger.trace("Item '{}' found", item);

				itemElement.click();
				return true;
			}

		logger.trace("Item '{}' not found", item);

		return false;
	}

}
